"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveAllLocalImports = void 0;
/**
 * Exported as `'resolve-import/resolve-all-local-imports'`
 * @module
 */
const path_1 = require("path");
const url_1 = require("url");
const errors_js_1 = require("./errors.js");
const file_exists_js_1 = require("./file-exists.js");
const find_dep_package_js_1 = require("./find-dep-package.js");
const read_pkg_js_1 = require("./read-pkg.js");
const resolve_all_exports_js_1 = require("./resolve-all-exports.js");
const resolve_conditional_value_js_1 = require("./resolve-conditional-value.js");
const resolve_import_js_1 = require("./resolve-import.js");
const star_glob_js_1 = require("./star-glob.js");
const to_file_url_js_1 = require("./to-file-url.js");
const to_path_js_1 = require("./to-path.js");
/**
 * Given a path or file URL to a package.json file, return an object where each
 * possible local import path is mapped to the file URL that it would resolve
 * to.
 *
 * Invalid and non-resolving imports are omitted.
 */
const resolveAllLocalImports = async (packageJsonPath, options = {}) => {
    const pjPath = (0, to_path_js_1.toPath)(packageJsonPath);
    const pjDir = (0, path_1.dirname)(pjPath);
    const pjURL = (0, to_file_url_js_1.toFileURL)(packageJsonPath);
    const pkg = await (0, read_pkg_js_1.readPkg)(pjPath);
    if (!pkg) {
        throw (0, errors_js_1.invalidPackage)(packageJsonPath, exports.resolveAllLocalImports);
    }
    const results = {};
    for (const [sub, target] of getNamedImportsList(pkg, options)) {
        // if the import is local, then look it up
        // if it's another package, then look up that package
        // if it's another package with a *, then look up all exports
        // of that package, and filter by the matches.
        const parts = target.match(/^(@[^\/]+\/[^\/]+|[^\/]+)/);
        // make internal package named modules consistently `./`
        const name = pkg.name;
        // non-matches already filtered out
        /* c8 ignore start */
        if (!parts)
            continue;
        /* c8 ignore stop */
        const ssub = sub.split('*');
        const starget = target.split('*');
        const star = ssub.length === 2 && starget.length === 2;
        if (!star) {
            // simple case, no * replacement
            // if not found, just omit it.
            // do a full resolve, because the target can be anything like
            // './foo/bar' or 'dep/blah', etc.
            try {
                results[sub] = await (0, resolve_import_js_1.resolveImport)(target, pjURL);
            }
            catch { }
            continue;
        }
        // has a star, have to glob if it's localPath, or look up exports if not
        const localPath = parts[1] === '.';
        if (localPath) {
            for (const [rep, target] of await (0, star_glob_js_1.starGlob)(starget, pjDir)) {
                results[ssub[0] + rep + ssub[1]] = (0, url_1.pathToFileURL)(target);
            }
            continue;
        }
        const localName = parts[1] === name;
        const dep = !localPath && !localName ? parts[1] : null;
        // if we can't find the package, it's not valid.
        const ppath = dep ? await (0, find_dep_package_js_1.findDepPackage)(dep, pjDir) : pjDir;
        if (!ppath)
            continue;
        const pj = (0, path_1.resolve)(ppath, 'package.json');
        if (!(await (0, file_exists_js_1.fileExists)(pj))) {
            continue;
        }
        const allExports = await (0, resolve_all_exports_js_1.resolveAllExports)(pj);
        for (const [k, v] of Object.entries(allExports)) {
            if (k === '.' || k === './')
                continue;
            const i = dep + k.substring(1);
            if (i.startsWith(starget[0]) && i.endsWith(starget[1])) {
                const s = ssub[0] +
                    i.substring(starget[0].length, i.length - starget[1].length) +
                    ssub[1];
                // should be impossible to throw, because we're pulling the list
                // from the package itself, and it gets resolved at that point.
                /* c8 ignore start */
                try {
                    results[s] = await (0, resolve_import_js_1.resolveImport)(v, pjURL);
                }
                catch { }
                /* c8 ignore stop */
            }
        }
    }
    return results;
};
exports.resolveAllLocalImports = resolveAllLocalImports;
/**
 * Get the condition-resolved targets of all imports
 *
 * Stars are not expanded.
 */
const getNamedImportsList = (pkg, options) => {
    const results = [];
    const { imports } = pkg;
    if (!imports || typeof imports !== 'object')
        return results;
    for (const [k, v] of Object.entries(imports)) {
        const r = (0, resolve_conditional_value_js_1.resolveConditionalValue)(v, options);
        if (r && !r.startsWith('#'))
            results.push([k, r]);
    }
    return results;
};
//# sourceMappingURL=resolve-all-local-imports.js.map