"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveAllExports = void 0;
/**
 * Exported as `'resolve-import/resolve-all-exports'`
 * @module
 */
const path_1 = require("path");
const url_1 = require("url");
const errors_js_1 = require("./errors.js");
const get_named_exports_list_js_1 = require("./get-named-exports-list.js");
const read_pkg_js_1 = require("./read-pkg.js");
const resolve_export_js_1 = require("./resolve-export.js");
const star_glob_js_1 = require("./star-glob.js");
const to_path_js_1 = require("./to-path.js");
/**
 * Given a path or file URL to a package.json file, return an object where each
 * possible export path is mapped to the file URL that it would resolve to.
 *
 * Invalid exports are omitted. No errors are raised as long as the file is a
 * valid `package.json`.
 *
 * Note: in cases like `"./x/*": "./file.js"`, where the list of possible
 * import paths is unbounded, the returned object will contain `"./x/*"` as the
 * key, since there's no way to expand that to every possible match.
 */
const resolveAllExports = async (packageJsonPath, options = {}) => {
    const pjPath = (0, to_path_js_1.toPath)(packageJsonPath);
    const pjDir = (0, path_1.dirname)(pjPath);
    const pkg = await (0, read_pkg_js_1.readPkg)(pjPath);
    if (!pkg) {
        throw (0, errors_js_1.invalidPackage)(packageJsonPath, exports.resolveAllExports);
    }
    const results = {};
    const { exports } = pkg;
    for (const sub of (0, get_named_exports_list_js_1.getNamedExportsList)(exports)) {
        let res;
        // this can't shouldn't be able to actually throw, because we're
        // pulling the list from the set itself.
        /* c8 ignore start */
        try {
            res = (0, resolve_export_js_1.resolveExport)(sub, exports, pjPath, pjPath, options);
        }
        catch { }
        if (!res)
            continue;
        /* c8 ignore stop */
        // if it contains a *, then we have to glob,
        // in package.json exports * is actually **, but only
        // relevant if there is exactly ONE star
        const sres = res.split('*');
        const ssub = sub.split('*');
        if (sres.length === 2 && ssub.length === 2) {
            for (const [rep, target] of await (0, star_glob_js_1.starGlob)(sres, pjDir)) {
                results[ssub[0] + rep + ssub[1]] = (0, url_1.pathToFileURL)(target);
            }
        }
        else {
            results[sub] = (0, url_1.pathToFileURL)((0, path_1.resolve)(pjDir, res));
        }
    }
    return results;
};
exports.resolveAllExports = resolveAllExports;
//# sourceMappingURL=resolve-all-exports.js.map