//
//  UIButton+Rx.swift
//  RxCocoa
//
//  Created by Krunoslav Zaher on 3/28/15.
//  Copyright © 2015 Krunoslav Zaher. All rights reserved.
//

#if os(iOS)

import RxSwift
import UIKit

extension Reactive where Base: UIButton {
    
    /// Reactive wrapper for `TouchUpInside` control event.
    public var tap: ControlEvent<Void> {
        return controlEvent(.touchUpInside)
    }
}

#endif

#if os(tvOS)

import RxSwift
import UIKit

extension Reactive where Base: UIButton {

    /// Reactive wrapper for `PrimaryActionTriggered` control event.
    public var primaryAction: ControlEvent<Void> {
        return controlEvent(.primaryActionTriggered)
    }

}

#endif

#if os(iOS) || os(tvOS)

import RxSwift
import UIKit

#if swift(>=4.2)
    public typealias UIControlState = UIControl.State
#endif

extension Reactive where Base: UIButton {
    
    /// Reactive wrapper for `setTitle(_:for:)`
    public func title(for controlState: UIControlState = []) -> Binder<String?> {
        return Binder(self.base) { (button, title) -> () in
            button.setTitle(title, for: controlState)
        }
    }

    /// Reactive wrapper for `setImage(_:for:)`
    public func image(for controlState: UIControlState = []) -> Binder<UIImage?> {
        return Binder(self.base) { (button, image) -> () in
            button.setImage(image, for: controlState)
        }
    }

    /// Reactive wrapper for `setBackgroundImage(_:for:)`
    public func backgroundImage(for controlState: UIControlState = []) -> Binder<UIImage?> {
        return Binder(self.base) { (button, image) -> () in
            button.setBackgroundImage(image, for: controlState)
        }
    }
    
}
#endif

#if os(iOS) || os(tvOS)

    import RxSwift
    import UIKit
    
    extension Reactive where Base: UIButton {
        
        /// Reactive wrapper for `setAttributedTitle(_:controlState:)`
        public func attributedTitle(for controlState: UIControlState = []) -> Binder<NSAttributedString?> {
            return Binder(self.base) { (button, attributedTitle) -> () in
                button.setAttributedTitle(attributedTitle, for: controlState)
            }
        }
        
    }
#endif
