//
//  RefCount.swift
//  Rx
//
//  Created by Krunoslav Zaher on 3/5/15.
//  Copyright (c) 2015 Krunoslav Zaher. All rights reserved.
//

import Foundation

class RefCount_<Element> : Sink<Element>, ObserverType {
    let parent: RefCount<Element>
    typealias ParentState = RefCount<Element>.State
    
    init(parent: RefCount<Element>, observer: ObserverOf<Element>, cancel: Disposable) {
        self.parent = parent
        super.init(observer: observer, cancel: cancel)
    }
    
    func run() -> Disposable {
        let subscription = self.parent.source.subscribe(self)
        
        let state = self.parent.state
        
        self.parent.lock.performLocked {
            if state.count == 0 {
                let disposable = self.parent.source.connect()
                self.parent.state.count = 1
                self.parent.state.connectableSubscription = disposable
            }
            else {
                self.parent.state.count = state.count + 1
            }
        }
        
        return AnonymousDisposable {
            subscription.dispose()
            self.parent.lock.performLocked {
                let state = self.parent.state
                if state.count == 1 {
                    state.connectableSubscription!.dispose()
                    self.parent.state.count = 0
                    self.parent.state.connectableSubscription = nil
                }
                else if state.count > 1 {
                    self.parent.state.count = state.count - 1
                }
                else {
                    rxFatalError("Something went wrong with RefCount disposing mechanism")
                }
            }
        }
    }

    func on(event: Event<Element>) {
        let observer = state.observer
        
        switch event {
        case .Next:
            observer.on(event)
        case .Error: fallthrough
        case .Completed:
            observer.on(event)
            self.dispose()
        }
    }
}

class RefCount<Element>: Producer<Element> {
    typealias State = (
        count: Int,
        connectableSubscription: Disposable?
    )
    
    var lock = Lock()
    
    var state: State = (
        count: 0,
        connectableSubscription: nil
    )
    
    let source: ConnectableObservableType<Element>
    
    init(source: ConnectableObservableType<Element>) {
        self.source = source
    }
    
    override func run(observer: ObserverOf<Element>, cancel: Disposable, setSink: (Disposable) -> Void) -> Disposable {
        let sink = RefCount_(parent: self, observer: observer, cancel: cancel)
        setSink(sink)
        return sink.run()
    }
}