//
//  LowEnergySupervisionTimeout.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/14/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

/// Supervision Timeout
///
/// Supervision timeout for the LE Link.
/// Range: 0x000A to 0x0C80
/// Time = N * 10 msec
/// Time Range: 100 msec to 32 seconds
///
/// - SeeAlso: [Vol 6] Part B, Section 4.5.2
@frozen
public struct LowEnergySupervisionTimeout: RawRepresentable, Equatable, Hashable, Comparable {

    /// 100 msec
    public static let min = LowEnergySupervisionTimeout(0x000A)

    /// 32 seconds
    public static let max = LowEnergySupervisionTimeout(0x0C80)

    public let rawValue: UInt16

    public init?(rawValue: UInt16) {

        guard rawValue >= LowEnergySupervisionTimeout.min.rawValue,
            rawValue <= LowEnergySupervisionTimeout.max.rawValue
        else { return nil }

        assert((LowEnergySupervisionTimeout.min.rawValue...LowEnergySupervisionTimeout.max.rawValue).contains(rawValue))

        self.rawValue = rawValue
    }

    /// Time = N * 10 msec
    public var miliseconds: Double {

        return Double(rawValue) * 10
    }

    // Private, unsafe
    private init(_ rawValue: UInt16) {
        self.rawValue = rawValue
    }

    // Comparable
    public static func < (lhs: LowEnergySupervisionTimeout, rhs: LowEnergySupervisionTimeout) -> Bool {

        return lhs.rawValue < rhs.rawValue
    }
}
