//
//  HCILESetResolvablePrivateAddressTimeout.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/14/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

import Foundation

// MARK: - Method

public extension BluetoothHostControllerInterface {

    /// LE Set Resolvable Private Address Timeout Command
    ///
    /// The command set the length of time the Controller uses a Resolvable Private Address before a new resolvable private address is generated and starts being used.
    func lowEnergySetResolvablePrivateAddressTimeout(rpaTimeout: HCILESetResolvablePrivateAddressTimeout.RPATimeout, timeout: HCICommandTimeout = .default) async throws {

        let parameters = HCILESetResolvablePrivateAddressTimeout(rpaTimeout: rpaTimeout)

        try await deviceRequest(parameters, timeout: timeout)
    }
}

// MARK: - Command

/// LE Set Resolvable Private Address Timeout Command
///
/// The command set the length of time the Controller uses a
/// Resolvable Private Address before a new resolvable private address is
/// generated and starts being used.
///
/// This timeout applies to all addresses generated by the Controller.
@frozen
public struct HCILESetResolvablePrivateAddressTimeout: HCICommandParameter {

    public static let command = HCILowEnergyCommand.setResolvablePrivateAddressTimeout  //0x002E

    /// Default: N= 0x0384 (900 s or 15 minutes)
    public static var defaultRpaTimeout: RPATimeout { RPATimeout(0x0384) }

    /// RPA_Timeout measured in s
    /// Range for N: 0x0001 – 0xA1B8 (1 s – approximately 11.5 hours)
    /// Default: N= 0x0384 (900 s or 15 minutes)
    public let rpaTimeout: RPATimeout  //RPA_Timeout

    public init(rpaTimeout: RPATimeout = Self.defaultRpaTimeout) {
        self.rpaTimeout = rpaTimeout
    }

    public var data: Data {

        let rpaTimeoutBytes = rpaTimeout.rawValue.littleEndian.bytes

        return Data([
            rpaTimeoutBytes.0,
            rpaTimeoutBytes.1
        ])
    }

    /// RPA_Timeout measured in s
    /// Range for N: 0x0001 – 0xA1B8 (1 s – approximately 11.5 hours)
    public struct RPATimeout: RawRepresentable, Equatable, Comparable, Hashable, Sendable {

        /// 2.5 msec
        public static var min: RPATimeout { RPATimeout(0x0001) }

        /// 10.24 seconds
        public static var max: RPATimeout { RPATimeout(0xA1B8) }

        public let rawValue: UInt16

        public init?(rawValue: UInt16) {

            guard rawValue >= RPATimeout.min.rawValue,
                rawValue <= RPATimeout.max.rawValue
            else { return nil }

            self.rawValue = rawValue
        }

        // Private, unsafe
        fileprivate init(_ rawValue: UInt16) {
            self.rawValue = rawValue
        }

        // Comparable
        public static func < (lhs: RPATimeout, rhs: RPATimeout) -> Bool {
            return lhs.rawValue < rhs.rawValue
        }

        public static func > (lhs: RPATimeout, rhs: RPATimeout) -> Bool {
            return lhs.rawValue > rhs.rawValue
        }
    }
}
