//
//  AdvertisingInterval.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/1/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

/// Advertising Interval
///
/// For all undirected advertising events, the time between the start of two consec- utive advertising events (`T_advEvent`) is computed as follows for each adver- tising event:
/// ```
/// T_advEvent = advInterval + advDelay
/// ```
///
/// The `advInterval` shall be an integer multiple of 0.625 ms in the range of `20` ms to `10.24` s. If the advertising event type is either a scannable undirected event type or a non-connectable undirected event type, the advInterval shall not be less than 100 ms. If the advertising event type is a connectable undirected event type, the advInterval can be 20 ms or greater.
///
/// The `advDelay` is a pseudo-random value with a range of 0 ms to 10 ms generated by the Link Layer for each advertising event.
///
/// As illustrated, the advertising events are perturbed in time using the `advDelay`.
///
/// ![Image](https://github.com/PureSwift/Bluetooth/raw/master/Assets/AdvertisingInterval.png)
@frozen
public struct AdvertisingInterval: Equatable, Hashable {

    public let rawValue: UInt16

    public init?(rawValue: UInt16) {
        guard rawValue <= AdvertisingInterval.max.rawValue,
            rawValue >= AdvertisingInterval.min.rawValue
        else { return nil }
        self.rawValue = rawValue
    }

    private init(_ unsafe: UInt16) {
        self.rawValue = unsafe
    }
}

// MARK: - Conversion

public extension AdvertisingInterval {

    /// Initialize with the specified time in miliseconds.
    init?(miliseconds: Int) {
        let rawValue = Double(miliseconds) / 0.625
        guard rawValue <= Double(AdvertisingInterval.max.rawValue),
            rawValue >= Double(AdvertisingInterval.min.rawValue)
        else { return nil }
        self.rawValue = UInt16(rawValue)
    }

    /// Value converted to miliseconds.
    var miliseconds: Int {
        return Int(Double(rawValue) * 0.625)
    }
}

// MARK: - Definitions

public extension AdvertisingInterval {

    /// Minimum value.
    static var min: AdvertisingInterval { return AdvertisingInterval(0x0020) }

    /// Maximum value.
    static var max: AdvertisingInterval { return AdvertisingInterval(0x4000) }

    /// Default value.
    static var `default`: AdvertisingInterval { return AdvertisingInterval(0x0800) }
}

// MARK: - Comparable

extension AdvertisingInterval: Comparable {

    public static func < (lhs: AdvertisingInterval, rhs: AdvertisingInterval) -> Bool {
        return lhs.rawValue < rhs.rawValue
    }
}

// MARK: - CustomStringConvertible

extension AdvertisingInterval: CustomStringConvertible {

    public var description: String {
        return "\(miliseconds)ms"
    }
}
