//
//  DarwinTests.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/1/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

import XCTest
import Foundation
@testable import Bluetooth
import BluetoothGAP

#if os(macOS)
import CoreBluetooth
import IOKit
import Darwin

final class DarwinTests: XCTestCase {
    
    // MARK: - Assigned Numbers
    
    func testGAPAppearance() {
        
        XCTAssertEqual(GAPAppearance.Unknown.unknown.rawValue, IOKitBluetoothGAPAppearance.Unknown.rawValue)
        XCTAssertEqual(GAPAppearance.Phone.generic.rawValue, IOKitBluetoothGAPAppearance.GenericPhone.rawValue)
        XCTAssertEqual(GAPAppearance.Computer.generic.rawValue, IOKitBluetoothGAPAppearance.GenericComputer.rawValue)
    }
    
    func testCompanyIdentifiers() {
        
        XCTAssertEqual(CompanyIdentifier.ericssonTechnologyLicensing.rawValue, IOKitBluetoothCompanyIdentifers.EricssonTechnologyLicensing.rawValue)
        XCTAssertEqual(CompanyIdentifier.nokiaMobilePhones.rawValue, IOKitBluetoothCompanyIdentifers.NokiaMobilePhones.rawValue)
        XCTAssertEqual(CompanyIdentifier.ibm.rawValue, IOKitBluetoothCompanyIdentifers.IBM.rawValue)
        XCTAssertEqual(CompanyIdentifier.intel.rawValue, IOKitBluetoothCompanyIdentifers.Intel.rawValue)
        XCTAssertEqual(CompanyIdentifier.apple.rawValue, IOKitBluetoothCompanyIdentifers.Apple.rawValue)
        XCTAssertEqual(CompanyIdentifier.google.rawValue, IOKitBluetoothCompanyIdentifers.Google.rawValue)
        XCTAssertEqual(CompanyIdentifier.microsoft.rawValue, IOKitBluetoothCompanyIdentifers.Microsoft.rawValue)
    }
    
    // MARK: - BluetoothUUID
    
    func testCoreBluetoothUUID() {
        
        do {
            
            let uuid = BluetoothUUID.bit16(0xFEA9)
            
            let coreBluetoothUUID = CBUUID(uuid)
            
            XCTAssert(coreBluetoothUUID.uuidString == uuid.rawValue)
            
            XCTAssert(uuid.bigEndian.data == coreBluetoothUUID.data, "\(uuid.data) == \(coreBluetoothUUID.data)")
        }
        
        do {
            
            let uuid = BluetoothUUID() // 128 bit
            
            let coreBluetoothUUID = CBUUID(uuid)
            
            XCTAssert(coreBluetoothUUID.uuidString == uuid.rawValue)
            
            XCTAssert(uuid.bigEndian.data == coreBluetoothUUID.data, "\(uuid.data) == \(coreBluetoothUUID.data)")
        }
        
        do {
            
            let coreBluetoothUUID = CBUUID(string: "FEA9")
            
            let uuid = BluetoothUUID(coreBluetoothUUID)
            
            XCTAssert(coreBluetoothUUID.uuidString == uuid.rawValue)
            
            XCTAssert(uuid.bigEndian.data == coreBluetoothUUID.data, "\(uuid.data) == \(coreBluetoothUUID.data)")
        }
        
        do {
            
            let coreBluetoothUUID = CBUUID(string: "68753A44-4D6F-1226-9C60-0050E4C00067")
            
            let uuid = BluetoothUUID(coreBluetoothUUID)
            
            XCTAssert(coreBluetoothUUID.uuidString == uuid.rawValue)
            
            XCTAssert(uuid.bigEndian.data == coreBluetoothUUID.data, "\(uuid.data) == \(coreBluetoothUUID.data)")
        }
    }
    
    func testCoreBluetoothPerfomanceStringParseUUID() {
        
        let uuids = randomUUIDs.map { $0.uuidString }
        
        measure { uuids.forEach { _ = CBUUID(string: $0) } }
    }
    
    func testCoreBluetoothPerfomanceStringUUID() {
        
        let uuids = randomUUIDs.map { CBUUID(nsuuid: $0) }
        
        measure { uuids.forEach { let _ = $0.uuidString } }
    }
    
    func testCoreBluetoothPerformanceDataParseUUID() {
        
        let uuids = randomUUIDs.map { $0.data }
        
        measure { uuids.forEach { _ = CBUUID(data: $0) } }
    }
    
    func testCoreBluetoothPerformanceDataUUID() {
        
        let uuids = randomUUIDs.map { CBUUID(nsuuid: $0) }
        
        measure { uuids.forEach { let _ = $0.data } }
    }
    
    // MARK: - Code Generators
    
    func testGenerateDefinedUUID() {
        
        let uuids = definedUUIDs.sorted(by: { $0.key < $1.key })
        
        var generatedCode = ""
        
        var memberNameCache = [UInt16: String]()
        
        func 🖨(_ text: String) {
            
            generatedCode += text + "\n"
        }
        
        let dateFormatter = DateFormatter()
        dateFormatter.dateStyle = .short
        dateFormatter.timeStyle = .none
        
        let fileDate = dateFormatter.string(from: Date())
        
        🖨("//")
        🖨("//  DefinedUUIDExtension.swift")
        🖨("//  Bluetooth")
        🖨("//")
        🖨("//  Generated by Alsey Coleman Miller on \(fileDate).")
        🖨("//")
        🖨("")
        🖨("public extension BluetoothUUID {")
        🖨("")
        
        for (uuidValue, name) in uuids {
            
            let uuid = BluetoothUUID.bit16(uuidValue)
            
            let sanitizedName = sanitize(name: name)
            
            let llamaCaseName = llamaCase(sanitizedName)
            
            var memberName = llamaCaseName
            
            // prevent duplicate entries
            var duplicateNumber = 1
            while memberNameCache.values.contains(memberName) {
                
                duplicateNumber += 1
                memberName = llamaCaseName + "\(duplicateNumber)"
            }
            
            let comment = name + " " + "(`0x\(uuid.rawValue)`)"
            
            🖨("    /// " + comment)
            🖨("    static var " + memberName + ": BluetoothUUID {")
            🖨("        return .bit16(0x" + uuid.rawValue + ")")
            🖨("    }")
            🖨("")
            
            memberNameCache[uuidValue] = memberName
        }
        
        🖨("}")
        
        var filename = NSTemporaryDirectory() + "DefinedUUIDExtension.swift"
        XCTAssertNoThrow(try generatedCode.write(toFile: filename, atomically: true, encoding: .utf8))
        
        print("Generated Swift code \(filename)")
        
        // generate unit test for extensions
        generatedCode = ""
        
        🖨("//")
        🖨("//  DefinedUUIDTests.swift")
        🖨("//  Bluetooth")
        🖨("//")
        🖨("//  Generated by Alsey Coleman Miller on \(fileDate).")
        🖨("//")
        🖨("")
        🖨("import XCTest")
        🖨("import Foundation")
        🖨("@testable import Bluetooth")
        🖨("")
        🖨("// swiftlint:disable type_body_length")
        🖨("final class DefinedUUIDTests: XCTestCase {")
        🖨("")
        🖨("    static let allTests = [")
        🖨("        (\"testDefinedUUID\", testDefinedUUID)")
        🖨("    ]")
        🖨("")
        🖨("    func testDefinedUUID() {")
        🖨("")
        
        
        // generate test methods
        
        for (uuidValue, name) in uuids {
            
            let uuid = BluetoothUUID.bit16(uuidValue)
            
            guard let memberName = memberNameCache[uuidValue]
                else { XCTFail("No extension generated for \(uuid)"); return }
            
            
            🖨("        /// \(name)")
            🖨("        XCTAssertEqual(BluetoothUUID.\(memberName).rawValue, \"\(uuid.rawValue)\")")
            🖨("        XCTAssertEqual(BluetoothUUID.\(memberName), .bit16(0x\(uuid.rawValue)))")
            🖨("        XCTAssertEqual(BluetoothUUID.\(memberName), .bit16(\(uuidValue)))")
            🖨("        XCTAssertEqual(BluetoothUUID.\(memberName).name, \"\(name)\")")
            🖨("        XCTAssertNotEqual(BluetoothUUID.\(memberName), .bit32(\(uuidValue)))")
            🖨("        XCTAssertNotEqual(BluetoothUUID.\(memberName), .bit32(0x\(uuid.rawValue)))")
            🖨("")
        }
        
        🖨("    }")
        🖨("")
        🖨("}")
        🖨("// swiftlint:enable type_body_length")
        
        filename = NSTemporaryDirectory() + "DefinedUUIDTests.swift"
        XCTAssertNoThrow(try generatedCode.write(toFile: filename, atomically: true, encoding: .utf8))
        
        print("Generated Swift code \(filename)")
    }
    
    func testGenerateUnitIdentifier() {
        
        let unitsMethodNames: [UInt16: String] = [
            0x2700: "unitless",
            0x2701: "metre",
            0x2702: "kilogram",
            0x2703: "second",
            0x2704: "ampere",
            0x2705: "kelvin",
            0x2706: "mole",
            0x2707: "candela",
            0x2710: "area",
            0x2711: "volume",
            0x2712: "velocity",
            0x2713: "acceleration",
            0x2714: "wavenumber",
            0x2715: "density",
            0x2716: "surfaceDensity",
            0x2717: "specificVolume",
            0x2718: "currentDensity",
            0x2719: "magneticFieldStrengh",
            0x271A: "amountConcentration",
            0x271B: "massConcentration",
            0x271C: "luminance",
            0x271D: "refractiveIndex",
            0x271E: "relativePermeability",
            0x2720: "planeAngle",
            0x2721: "solidAngle",
            0x2722: "frequency",
            0x2723: "force",
            0x2724: "pascalPressure",
            0x2725: "energy",
            0x2726: "power",
            0x2727: "coulomb",
            0x2728: "electricPotential",
            0x2729: "capitance",
            0x272A: "electricResistance",
            0x272B: "electricConductance",
            0x272C: "magneticFlux",
            0x272D: "magneticFluxDensity",
            0x272E: "inductance",
            0x272F: "celsius",
            0x2730: "luminousFlux",
            0x2731: "illuminance",
            0x2732: "becquerel",
            0x2733: "absorbedDose",
            0x2734: "sievert",
            0x2735: "katal",
            0x2740: "pascalSecond",
            0x2741: "newtonMetre",
            0x2742: "surfaceTension",
            0x2743: "angularVelocity",
            0x2744: "angularAcceleration",
            0x2745: "heatFluxDensity",
            0x2746: "heatCapacity",
            0x2747: "specificHeatCapacity",
            0x2748: "specificEnergy",
            0x2749: "thermalConductivity",
            0x274A: "energyDensity",
            0x274B: "electricFieldStrength",
            0x274C: "electricChargeDensity",
            0x274D: "surfaceChargeDensity",
            0x274E: "electricFluxDensity",
            0x274F: "permittivity",
            0x2750: "permeability",
            0x2751: "molarEnergy",
            0x2752: "molarEntropy",
            0x2753: "exposure",
            0x2754: "absorbedDoseRate",
            0x2755: "radradiantIntensityiance",
            0x2756: "radiance",
            0x2757: "catalyticActivity",
            0x2760: "minute",
            0x2761: "hour",
            0x2762: "day",
            0x2763: "degree",
            0x2764: "planeAngleMinute",
            0x2765: "planeAngleSecond",
            0x2766: "hectare",
            0x2767: "litre",
            0x2768: "tonne",
            0x2780: "bar",
            0x2781: "millimetreOfMercury",
            0x2782: "ngstrm",
            0x2783: "nauticalMile",
            0x2784: "barn",
            0x2785: "velocityKnot",
            0x2786: "neper",
            0x2787: "bel",
            0x27A0: "yard",
            0x27A1: "parsec",
            0x27A2: "inch",
            0x27A3: "foot",
            0x27A4: "mile",
            0x27A5: "pressurePoundForce",
            0x27A6: "kilometrePerHour",
            0x27A7: "milePerHour",
            0x27A8: "revolutionPerMinute",
            0x27A9: "gramCalorie",
            0x27AA: "kilogramCalorie",
            0x27AB: "kilowattHour",
            0x27AC: "degreeFahrenheit",
            0x27AD: "percentage",
            0x27AE: "perMille",
            0x27AF: "beatsPerMinute",
            0x27B0: "ampereHours",
            0x27B1: "milligramPerDecilitre",
            0x27B2: "millimolePerLitre",
            0x27B3: "year",
            0x27B4: "month",
            0x27B5: "concentration",
            0x27B6: "irrandiance",
            0x27B7: "millilitre",
            0x27B8: "pound",
            0x27B9: "metabolicEquivalent",
            0x27BA: "step",
            0x27BC: "stroke",
            0x27BD: "pace",
            0x27BE: "luminousEfficacy",
            0x27BF: "luminousEnergy",
            0x27C0: "luminousExposure",
            0x27C1: "massFlow",
            0x27C2: "volumeFlow"
        ]
        
        var generatedCode = ""
        
        var memberNameCache = [UInt16: String]()
        
        func 🖨(_ text: String) {
            generatedCode += text + "\n"
        }
        
        let dateFormatter = DateFormatter()
        dateFormatter.dateStyle = .short
        dateFormatter.timeStyle = .none
        
        let fileDate = dateFormatter.string(from: Date())
        
        🖨("//")
        🖨("//  UnitIdentifierExtension.swift")
        🖨("//  Bluetooth")
        🖨("//")
        🖨("")
        🖨("public extension UnitIdentifier {")
        🖨("")
        
        for (identifier, unit) in units.sorted(by: { $0.key < $1.key }) {
            
            var memberName = unitsMethodNames[identifier]!
            
            // prevent duplicate entries
            var duplicateNumber = 1
            while memberNameCache.values.contains(memberName) {
                
                duplicateNumber += 1
                memberName = memberName + "\(duplicateNumber)"
            }
            
            let hexValue = "0x\(identifier.toHexadecimal())"
            let comment = unit.name + " " + "(`\(hexValue)`)"
            
            🖨("    /// " + comment)
            🖨("    static var " + memberName + ": UnitIdentifier {")
            🖨("        return UnitIdentifier(rawValue: \(hexValue))")
            🖨("    }")
            🖨("")
            
            memberNameCache[identifier] = memberName
        }
        
        🖨("}")
        
        var filename = NSTemporaryDirectory() + "UnitIdentifierExtension.swift"
        XCTAssertNoThrow(try generatedCode.write(toFile: filename, atomically: true, encoding: .utf8))
        
        print("Generated Swift code \(filename)")
        
        // generate unit test for extensions
        generatedCode = ""
        
        🖨("//")
        🖨("//  UnitIdentifierTests.swift")
        🖨("//  Bluetooth")
        🖨("//")
        🖨("//  Generated by Carlos Duclos on \(fileDate).")
        🖨("//")
        🖨("")
        🖨("import XCTest")
        🖨("import Foundation")
        🖨("@testable import Bluetooth")
        🖨("")
        🖨("// swiftlint:disable type_body_length")
        🖨("final class UnitIdentifierTests: XCTestCase {")
        🖨("")
        🖨("    static let allTests = [")
        🖨("        (\"testUnits\", testUnits)")
        🖨("    ]")
        🖨("")
        🖨("    func testUnits() {")
        🖨("")
        
        // generate test methods
        
        for (identifier, unit) in units.sorted(by: { $0.key < $1.key }) {
            
            guard let memberName = memberNameCache[identifier]
                else { XCTFail("No extension generated for \(identifier)"); return }
            
            let stringLiteral = unit.name.replacingOccurrences(of: "\"", with: "\\\"")
            let hexValue = "0x\(identifier.toHexadecimal())"
            
            🖨("        /// \(unit.name)")
            🖨("        XCTAssertEqual(UnitIdentifier.\(memberName).rawValue, \(hexValue))")
            🖨("        XCTAssertEqual(UnitIdentifier.\(memberName).type, \"\(unit.type)\")")
            🖨("        XCTAssertEqual(UnitIdentifier.\(memberName).name, \"\(stringLiteral)\")")
            🖨("        XCTAssertEqual(UnitIdentifier.\(memberName).description, \"\(identifier.toHexadecimal()) (\(stringLiteral))\")")
            🖨("")
        }
        
        🖨("    }")
        🖨("")
        🖨("}")
        🖨("// swiftlint:enable type_body_length")
        
        filename = NSTemporaryDirectory() + "UnitIdentifierTests.swift"
        XCTAssertNoThrow(try generatedCode.write(toFile: filename, atomically: true, encoding: .utf8))
        
        print("Generated Swift code \(filename)")
    }
}

// MARK: - Utilities

// https://gist.github.com/AmitaiB/bbfcba3a21411ee6d3f972320bcd1ecd
func camelCase(_ string: String) -> String {
    return string.components(separatedBy: CharacterSet.alphanumerics.inverted)
        .filter { !$0.isEmpty }
        .map { $0.capitalized }
        .joined()
}

func llamaCase(_ string: String) -> String {
    var result = camelCase(string)
    if let firstLetterCharacter = result.first {
        result = String(result.dropFirst())
        let firstLetter = String(firstLetterCharacter)
        result = firstLetter.lowercased() + result
    }
    return result
}

func uppercaseFirstLetter(_ string: String) -> String {
    var result = string
    if let firstLetterCharacter = result.first {
        result = String(result.dropFirst())
        let firstLetter = String(firstLetterCharacter)
        result = firstLetter.uppercased() + result
    }
    return result
}

func sanitize(name: String) -> String {
    
    let blackList = ["ASSA ABLOY"]
    
    guard blackList.contains(name) == false
        else { return name }
    
    var name = name
        .replacingOccurrences(of: "LLC \"", with: "")
        .replacingOccurrences(of: "\"", with: "")
        .replacingOccurrences(of: "3D ", with: "uuid3D")
        .replacingOccurrences(of: "IF, LLC", with: "ifLLC")
        .replacingOccurrences(of: "WHERE, Inc.", with: "whereInc")
        .replacingOccurrences(of: "Amazon.com Services, Inc.", with: "Amazon")
        .replacingOccurrences(of: ", Ltd. (QTIL)", with: "")
        .replacingOccurrences(of: "The ", with: "")
        .replacingOccurrences(of: "A/V", with: "av")
        .replacingOccurrences(of: " Incorporated", with: "")
        .replacingOccurrences(of: " Corporation", with: "")
        .replacingOccurrences(of: " Limited", with: "")
        .replacingOccurrences(of: " Pvt.", with: "")
        .replacingOccurrences(of: "GmbH & Co. KG", with: "")
        .replacingOccurrences(of: "GmbH & Co KG", with: "")
        .replacingOccurrences(of: "AG & Co. KGaA", with: "")
        .replacingOccurrences(of: "AG & Co. KG", with: "")
        .replacingOccurrences(of: "AG & Co.", with: "")
        .replacingOccurrences(of: " Corp.", with: "")
        .replacingOccurrences(of: " Corp", with: "")
        .replacingOccurrences(of: "Co.,Ltd", with: "")
        .replacingOccurrences(of: ",Co.Ltd", with: "")
        .replacingOccurrences(of: "CO.,LTD.", with: "")
        .replacingOccurrences(of: "Co.,", with: "")
        .replacingOccurrences(of: "Co.", with: "")
        .replacingOccurrences(of: " Sp. z o.o.", with: "")
        .replacingOccurrences(of: " ASA", with: "")
        .replacingOccurrences(of: " AS", with: "")
        .replacingOccurrences(of: " SA", with: "")
        .replacingOccurrences(of: " AB", with: "")
        .replacingOccurrences(of: " BV", with: "")
        .replacingOccurrences(of: " AG", with: "")
        .replacingOccurrences(of: " d.o.o.", with: "")
        .replacingOccurrences(of: " D.O.O.", with: "")
        .replacingOccurrences(of: " Oy", with: "")
        .replacingOccurrences(of: " gmbh", with: "")
        .replacingOccurrences(of: " GmbH", with: "")
        .replacingOccurrences(of: " B.V.", with: "")
        .replacingOccurrences(of: " b.v.", with: "")
        .replacingOccurrences(of: ",Inc.", with: "")
        .replacingOccurrences(of: ", inc.", with: "")
        .replacingOccurrences(of: " Inc", with: "")
        .replacingOccurrences(of: " INC", with: "")
        .replacingOccurrences(of: " LLC", with: "")
        .replacingOccurrences(of: " LTD", with: "")
        .replacingOccurrences(of: " Ltd", with: "")
        .replacingOccurrences(of: " ltd", with: "")
        .replacingOccurrences(of: " Ltd", with: "")
        .replacingOccurrences(of: " A/S", with: "")
        .replacingOccurrences(of: " S.A.", with: "")
        .replacingOccurrences(of: " S.L.", with: "")
        .replacingOccurrences(of: " ApS", with: "")
        .replacingOccurrences(of: " s.r.o.", with: "")
        .replacingOccurrences(of: " Srl", with: "")
    
    // if first letter is a number, add prefix
    if let firstCharacter = name.first,
        let _ = Int(String(firstCharacter)) {
        
        name = "uuid" + name
    }
    
    return name
}

#endif

/*
 BluetoothCompanyIdentifers
 
 The "living document" can be found on the Bluetooth SIG website:
 https://www.bluetooth.com/specifications/assigned-numbers/company-identifiers
 */

enum IOKitBluetoothCompanyIdentifers: UInt16
{
    case EricssonTechnologyLicensing                        =    0
    case NokiaMobilePhones                                    =    1
    case Intel                                                =    2
    case IBM                                                =    3
    case Toshiba                                            =    4
    case _3Com                                                =    5
    case Microsoft                                            =    6
    case Lucent                                            =    7
    case Motorola                                            =    8
    case InfineonTechnologiesAG                            =    9
    case CambridgeSiliconRadio                                =    10
    case SiliconWave                                        =    11
    case DigianswerAS                                        =    12
    case TexasInstruments                                    =    13
    case ParthusTechnologies                                =    14
    case Broadcom                                            =    15
    case MitelSemiconductor                                =    16
    case Widcomm                                            =    17
    case Zeevo                                                =    18
    case Atmel                                                =    19
    case MistubishiElectric                                =    20
    case RTXTelecom                                        =    21
    case KCTechnology                                        =    22
    case Newlogic                                            =    23
    case Transilica                                        =    24
    case RohdeandSchwarz                                    =    25
    case TTPCom                                            =    26
    case SigniaTechnologies                                =    27
    case ConexantSystems                                    =    28
    case Qualcomm                                            =    29
    case Inventel                                            =    30
    case AVMBerlin                                            =    31
    case Bandspeed                                            =    32
    case Mansella                                            =    33
    case NEC                                                =    34
    case WavePlusTechnology                                =    35
    case Alcatel                                            =    36
    case PhilipsSemiconductor                                =    37
    case CTechnologies                                        =    38
    case OpenInterface                                        =    39
    case RFCMicroDevices                                    =    40
    case Hitachi                                            =    41
    case SymbolTechnologies                                =    42
    case Tenovis                                            =    43
    case MacronixInternational                                =    44
    case GCTSemiconductor                                    =    45
    case NorwoodSystems                                    =    46
    case MewTelTechnology                                    =    47
    case STMicroelectronics                                =    48
    case Synopsys                                            =    49
    case RedMCommunications                                =    50
    case Commil                                            =    51
    case CATC                                                =    52
    case Eclipse                                            =    53
    case RenesasTechnology                                    =    54
    case Mobilian                                            =    55
    case Terax                                                =    56
    case IntegratedSystemSolution                            =    57
    case MatsushitaElectricIndustrial                        =    58
    case Gennum                                            =    59
    case ResearchInMotion                                    =    60
    case IPextreme                                            =    61
    case SystemsAndChips                                    =    62
    case BluetoothSIG                                        =    63
    case SeikoEpson                                        =    64
    case IntegratedSiliconSolution                            =    65
    case CONWISETechnology                                    =    66
    case ParrotSA                                            =    67
    case SocketCommunications                                =    68
    case AtherosCommunications                                =    69
    case MediaTek                                            =    70
    case Bluegiga                                            =    71
    case MarvellTechnologyGroup                            =    72
    case _3DSP                                                =    73
    case AccelSemiconductor                                =    74
    case ContinentialAutomotiveSystems                        =    75
    case Apple                                                =    76
    case StaccatoCommunications                            =    77
    case AvagoTechnologies                                    =    78
    case APT                                                =    79
    case SiRFTechnology                                    =    80
    case TZeroTechnologies                                    =    81
    case JandM                                                =    82
    case Free2Move                                            =    83
    case _3DiJoy                                            =    84
    case Plantronics                                        =    85
    case SonyEricssonMobileCommunications                    =    86
    case HarmonInternational                                =    87
    case Visio                                                =    88
    case NordicSemiconductor                                =    89
    case EMMicroElectronicMarin                            =    90
    case RalinkTechnology                                    =    91
    case BelkinInternational                                =    92
    case RealtekSemiconductor                                =    93
    case StonestreetOne                                    =    94
    case Wicentric                                            =    95
    case RivieraWaves                                        =    96
    case RDAMicroelectronics                                =    97
    case GibsonGuitars                                        =    98
    case MiCommand                                            =    99
    case BandXIInternational                                =    100
    case HewlettPackard                                    =    101
    case _9SolutionsOy                                        =    102
    case GNNetcom                                            =    103
    case GeneralMotors                                        =    104
    case AAndDEngineering                                    =    105
    case MindTree                                            =    106
    case PolarElectroOY                                    =    107
    case BeautifulEnterprise                                =    108
    case BriarTek                                            =    109
    case SummitDataCommunications                            =    110
    case SoundID                                            =    111
    case Monster                                            =    112
    case ConnectBlueAB                                        =    113
    case ShangHaiSuperSmartElectronics                        =    114
    case GroupSense                                        =    115
    case Zomm                                                =    116
    case SamsungElectronics                                =    117
    case CreativeTechnology                                =    118
    case LairdTechnologies                                    =    119
    case Nike                                                =    120
    case LessWire                                            =    121
    case MStarTechnologies                                    =    122
    case HanlynnTechnologies                                =    123
    case AAndRCambridge                                    =    124
    case SeersTechnology                                    =    125
    case SportsTrackingTechnologies                        =    126
    case AutonetMobile                                        =    127
    case DeLormePublishingCompany                            =    128
    case WuXiVimicro                                        =    129
    case SennheiserCommunications                            =    130
    case TimeKeepingSystems                                =    131
    case LudusHelsinki                                        =    132
    case BlueRadios                                        =    133
    case Equinux                                            =    134
    case GarminInternational                                =    135
    case Ecotest                                            =    136
    case GNResound                                            =    137
    case Jawbone                                            =    138
    case TopconPositioningSystems                            =    139
    case Gimbal                                            =    140
    case ZscanSoftware                                        =    141
    case Quintic                                            =    142
    case TelitWirelessSolutions                            =    143
    case FunaiElectric                                        =    144
    case AdvancedPANMOBILSystems                            =    145
    case ThinkOptics                                        =    146
    case UniversalElectriconics                            =    147
    case AirohaTechnology                                    =    148
    case NECLightning                                        =    149
    case ODMTechnology                                        =    150
    case ConnecteDevice                                    =    151
    case Zero1TV                                            =    152
    case ITechDynamicGlobalDistribution                    =    153
    case Alpwise                                            =    154
    case JiangsuToppowerAutomotiveElectronics                =    155
    case Colorfy                                            =    156
    case Geoforce                                            =    157
    case Bose                                                =    158
    case SuuntoOy                                            =    159
    case KensingtonComputerProductsGroup                    =    160
    case SRMedizinelektronik                                =    161
    case Vertu                                                =    162
    case MetaWatch                                            =    163
    case Linak                                                =    164
    case OTLDynamics                                        =    165
    case PandaOcean                                        =    166
    case Visteon                                            =    167
    case ARPDevicesUnlimited                                =    168
    case MagnetiMarelli                                    =    169
    case CaenRFID                                            =    170
    case IngenieurSystemgruppeZahn                            =    171
    case GreenThrottleGames                                =    172
    case PeterSystemtechnik                                =    173
    case Omegawave                                            =    174
    case Cinetix                                            =    175
    case PassifSemiconductor                                =    176
    case SarisCyclingGroup                                    =    177
    case Bekey                                                =    178
    case ClarinoxTechnologies                                =    179
    case BDETechnology                                        =    180
    case SwirlNetworks                                        =    181
    case MesoInternational                                    =    182
    case TreLab                                            =    183
    case QualcommInnovationCenter                            =    184
    case JohnsonControls                                    =    185
    case StarkeyLaboratories                                =    186
    case SPowerElectronics                                    =    187
    case AceSensor                                            =    188
    case Aplix                                                =    189
    case AAMPofAmerica                                        =    190
    case StalmartTechnology                                =    191
    case AMICCOMElectronics                                =    192
    case ShenzhenExcelsecuDataTechnology                    =    193
    case Geneq                                                =    194
    case Adidas                                            =    195
    case LGElectronics                                        =    196
    case OnsetComputer                                        =    197
    case SelflyBV                                            =    198
    case Quupa                                                =    199
    case GeLo                                                =    200
    case Evluma                                            =    201
    case MC10                                                =    202
    case BinauricSE                                        =    203
    case BeatsElectronics                                    =    204
    case MicrochipTechnology                                =    205
    case ElgatoSystems                                        =    206
    case ARCHOS                                            =    207
    case Dexcom                                            =    208
    case PolarElectroEurope                                =    209
    case DialogSemiconductor                                =    210
    case TaixingbangTechnology                                =    211
    case Kawantech                                            =    212
    case AustcoCommunicationsSystems                        =    213
    case TimexGroup                                        =    214
    case QualcommTechnologies                                =    215
    case QualcommConnectedExperiences                        =    216
    case VoyetraTurtleBeach                                =    217
    case txtrGMBH                                            =    218
    case Biosentronics                                        =    219
    case ProctorAndGamble                                    =    220
    case Hosiden                                            =    221
    case Musik                                                =    222
    case MisfitWearables                                    =    223
    case Google                                            =    224
    case Danlers                                            =    225
    case Semilink                                            =    226
    case InMusicBrands                                        =    227
    case LSResearch                                        =    228
    case EdenSoftwareConsultants                            =    229
    case Freshtemp                                            =    230
    case KSTechnologies                                    =    231
    case ACTSTechnologies                                    =    232
    case VtrackSystems                                        =    233
    case NielsenKellerman                                    =    234
    case ServerTechnology                                    =    235
    case BioResearchAssociates                                =    236
    case JollyLogic                                        =    237
    case AboveAverageOutcomes                                =    238
    case Bitsplitters                                        =    239
    case PayPal                                            =    240
    case WitronTechnology                                    =    241
    case MorseProject                                        =    242
    case KentDisplays                                        =    243
    case Nautilus                                            =    244
    case Smartifier                                        =    245
    case Elcometer                                            =    246
    case VSNTechnologies                                    =    247
    case AceUni                                            =    248
    case StickNFind                                        =    249
    case CrystalCode                                        =    250
    case KOUKAMM                                            =    251
    case Delphi                                            =    252
    case ValenceTech                                        =    253
    case StanleyBlackAndDecker                                =    254
    case TypeProducts                                        =    255
    case TomTomInternational                                =    256
    case FuGoo                                                =    257
    case Keiser                                            =    258
    case BangAndOlufson                                    =    259
    case PLUSLocationSystems                                =    260
    case UbiquitousComputingTechnology                        =    261
    case InnovativeYachtterSolutions                        =    262
    case WilliamDemantHolding                                =    263
    
    case InteropIdentifier                                    =    65535
};

enum IOKitBluetoothGAPAppearance: UInt16 {
    
    case Unknown = 0
    case GenericPhone = 64
    case GenericComputer = 128
    case GenericWatch = 192
    case GenericClock = 256
    case GenericDisplay = 320
    case GenericRemoteControl = 384
    case GenericEyeGlasses = 448
    case GenericTag = 512
    case GenericKeyring = 576
    case GenericMediaPlayer = 640
    case GenericBarcodeScanner = 704
    case GenericThermometer = 768
    case GenericHeartrateSensor = 832
    case GenericBloodPressure = 896
    case GenericHumanInterfaceDevice = 960
    case HumanInterfaceDeviceKeyboard = 961
    case HumanInterfaceDeviceMouse = 962
    case HumanInterfaceDeviceJoystick = 963
    case HumanInterfaceDeviceGamepad = 964
    case HumanInterfaceDeviceDigitizerTablet = 965
    case HumanInterfaceDeviceCardReader = 966
    case HumanInterfaceDeviceDigitalPen = 967
    case HumanInterfaceDeviceBarcodeScanner = 968
    case GenericGlucoseMeter = 1024
    case GenericRunningWalkingSensor = 1088
    case GenericCycling = 1152
}
