//
//  HCILESetResolvablePrivateAddressTimeout.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/14/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

import Foundation

// MARK: - Method

public extension BluetoothHostControllerInterface {
    
    /// LE Set Resolvable Private Address Timeout Command
    ///
    /// The command set the length of time the Controller uses a Resolvable Private Address before a new resolvable private address is generated and starts being used.
    func lowEnergySetResolvablePrivateAddressTimeout(rpaTimeout: HCILESetResolvablePrivateAddressTimeout.RPATimeout, timeout: HCICommandTimeout = .default) throws {
        
        let parameters = HCILESetResolvablePrivateAddressTimeout(rpaTimeout: rpaTimeout)
        
        try deviceRequest(parameters, timeout: timeout)
    }
}

// MARK: - Command

/// LE Set Resolvable Private Address Timeout Command
///
/// The command set the length of time the Controller uses a
/// Resolvable Private Address before a new resolvable private address is
/// generated and starts being used.
///
/// This timeout applies to all addresses generated by the Controller.
public struct HCILESetResolvablePrivateAddressTimeout: HCICommandParameter {
    
    public static let command = HCILowEnergyCommand.setResolvablePrivateAddressTimeout //0x002E
    
    /// Default: N= 0x0384 (900 s or 15 minutes)
    public static let defaultRpaTimeout = RPATimeout(0x0384)
    
    /// RPA_Timeout measured in s
    /// Range for N: 0x0001 – 0xA1B8 (1 s – approximately 11.5 hours)
    /// Default: N= 0x0384 (900 s or 15 minutes)
    public let rpaTimeout: RPATimeout //RPA_Timeout
    
    public init(rpaTimeout: RPATimeout = defaultRpaTimeout) {
        self.rpaTimeout = rpaTimeout
    }
    
    public var data: Data {
        
        let rpaTimeoutBytes = rpaTimeout.rawValue.littleEndian.bytes
        
        return Data([rpaTimeoutBytes.0,
                     rpaTimeoutBytes.1])
    }
    
    /// RPA_Timeout measured in s
    /// Range for N: 0x0001 – 0xA1B8 (1 s – approximately 11.5 hours)
    public struct RPATimeout: RawRepresentable, Equatable, Comparable, Hashable {
        
        /// 2.5 msec
        public static let min = RPATimeout(0x0001)
        
        /// 10.24 seconds
        public static let max = RPATimeout(0xA1B8)
        
        public let rawValue: UInt16
        
        public init?(rawValue: UInt16) {
            
            guard rawValue >= RPATimeout.min.rawValue,
                rawValue <= RPATimeout.max.rawValue
                else { return nil }
            
            self.rawValue = rawValue
        }
        
        // Private, unsafe
        fileprivate init(_ rawValue: UInt16) {
            self.rawValue = rawValue
        }
        
        // Equatable
        public static func == (lhs: RPATimeout, rhs: RPATimeout) -> Bool {
            
            return lhs.rawValue == rhs.rawValue
        }
        
        // Comparable
        public static func < (lhs: RPATimeout, rhs: RPATimeout) -> Bool {
            
            return lhs.rawValue < rhs.rawValue
        }
        
        // Hashable
        public var hashValue: Int {
            
            return Int(rawValue)
        }
    }
}
