//
//  AdvertisingInterval.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 6/1/18.
//  Copyright © 2018 PureSwift. All rights reserved.
//

/**
 Advertising Interval
 
 For all undirected advertising events, the time between the start of two consec- utive advertising events (`T_advEvent`) is computed as follows for each adver- tising event:
 ```
 T_advEvent = advInterval + advDelay
 ```
 
 The `advInterval` shall be an integer multiple of 0.625 ms in the range of `20` ms to `10.24` s. If the advertising event type is either a scannable undirected event type or a non-connectable undirected event type, the advInterval shall not be less than 100 ms. If the advertising event type is a connectable undirected event type, the advInterval can be 20 ms or greater.
 
 The `advDelay` is a pseudo-random value with a range of 0 ms to 10 ms generated by the Link Layer for each advertising event.
 
 As illustrated, the advertising events are perturbed in time using the `advDelay`.
 
 ![Image](https://github.com/PureSwift/Bluetooth/raw/master/Assets/AdvertisingInterval.png)
 */
public struct AdvertisingInterval {
    
    public let rawValue: UInt16
    
    public static let min = AdvertisingInterval(0x0020)
    
    public static let max = AdvertisingInterval(0x4000)
    
    public static let `default` = AdvertisingInterval(0x0800)
    
    public init?(rawValue: UInt16) {
        
        guard rawValue <= AdvertisingInterval.max.rawValue,
            rawValue >= AdvertisingInterval.min.rawValue
            else { return nil }
        
        self.rawValue = rawValue
    }
    
    private init(_ unsafe: UInt16) {
        
        self.rawValue = unsafe
    }
    
    public var miliseconds: Double {
        
        return Double(rawValue) * 0.625
    }
}

// MARK: - Equatable

extension AdvertisingInterval: Equatable {
    
    public static func == (lhs: AdvertisingInterval, rhs: AdvertisingInterval) -> Bool {
        
        return lhs.rawValue == rhs.rawValue
    }
}

// MARK: - Hashable

extension AdvertisingInterval: Hashable {
    
    public var hashValue: Int {
        
        return Int(rawValue)
    }
}

// MARK: - Comparable

extension AdvertisingInterval: Comparable {
    
    public static func < (lhs: AdvertisingInterval, rhs: AdvertisingInterval) -> Bool {
        
        return lhs.rawValue < rhs.rawValue
    }
}

// MARK: - CustomStringConvertible

extension AdvertisingInterval: CustomStringConvertible {
    
    public var description: String {
        
        return "\(miliseconds)ms"
    }
}
