//
//  DefinedUUID.swift
//  Bluetooth
//
//  Created by Alsey Coleman Miller on 1/5/18.
//

public extension BluetoothUUID {
    
    /// Bluetooth UUID Definition name.
    ///
    /// The Bluetooth SIG allocates a 16-bit Universally Unique Identifier (UUID) for use with
    /// a custom GATT-based service defined by the member, as well as a 16-bit Universally Unique Identifier (UUID)
    /// assignment for Standards Development Organizations (SDOs).
    ///
    /// [16-bit UUID Assigned Numbers](https://www.bluetooth.com/specifications/assigned-numbers/16-bit-uuids-for-members)
    /// [16 Bit UUIDs For SDOs](https://www.bluetooth.com/specifications/assigned-numbers/16-bit-uuids-for-sdos)
    public var name: String? {
        
        switch self {
        case let .bit16(value): return definedUUIDs[value]
        case .bit32, .bit128: return nil
        }
    }
}

/// https://www.bluetooth.com/specifications/assigned-numbers/16-bit-uuids-for-members
internal let definedUUIDs: [UInt16: String] = [
    
    0x0001: "SDP",
    0x0003: "RFCOMM",
    0x0005: "TCS-BIN",
    0x0007: "ATT",
    0x0008: "OBEX",
    0x000f: "BNEP",
    0x0010: "UPNP",
    0x0011: "HIDP",
    0x0012: "Hardcopy Control Channel",
    0x0014: "Hardcopy Data Channel",
    0x0016: "Hardcopy Notification",
    0x0017: "AVCTP",
    0x0019: "AVDTP",
    0x001b: "CMTP",
    0x001e: "MCAP Control Channel",
    0x001f: "MCAP Data Channel",
    0x0100: "L2CAP",
    /* 0x0101 to 0x0fff undefined */
    0x1000: "Service Discovery Server Service Class",
    0x1001: "Browse Group Descriptor Service Class",
    0x1002: "Public Browse Root",
    /* 0x1003 to 0x1100 undefined */
    0x1101: "Serial Port",
    0x1102: "LAN Access Using PPP",
    0x1103: "Dialup Networking",
    0x1104: "IrMC Sync",
    0x1105: "OBEX Object Push",
    0x1106: "OBEX File Transfer",
    0x1107: "IrMC Sync Command",
    0x1108: "Headset",
    0x1109: "Cordless Telephony",
    0x110a: "Audio Source",
    0x110b: "Audio Sink",
    0x110c: "A/V Remote Control Target",
    0x110d: "Advanced Audio Distribution",
    0x110e: "A/V Remote Control",
    0x110f: "A/V Remote Control Controller",
    0x1110: "Intercom",
    0x1111: "Fax",
    0x1112: "Headset AG",
    0x1113: "WAP",
    0x1114: "WAP Client",
    0x1115: "PANU",
    0x1116: "NAP",
    0x1117: "GN",
    0x1118: "Direct Printing",
    0x1119: "Reference Printing",
    0x111a: "Basic Imaging Profile",
    0x111b: "Imaging Responder",
    0x111c: "Imaging Automatic Archive",
    0x111d: "Imaging Referenced Objects",
    0x111e: "Handsfree",
    0x111f: "Handsfree Audio Gateway",
    0x1120: "Direct Printing Refrence Objects Service",
    0x1121: "Reflected UI",
    0x1122: "Basic Printing",
    0x1123: "Printing Status",
    0x1124: "Human Interface Device Service",
    0x1125: "Hardcopy Cable Replacement",
    0x1126: "HCR Print",
    0x1127: "HCR Scan",
    0x1128: "Common ISDN Access",
    /* 0x1129 and 0x112a undefined */
    0x112d: "SIM Access",
    0x112e: "Phonebook Access Client",
    0x112f: "Phonebook Access Server",
    0x1130: "Phonebook Access",
    0x1131: "Headset HS",
    0x1132: "Message Access Server",
    0x1133: "Message Notification Server",
    0x1134: "Message Access Profile",
    0x1135: "GNSS",
    0x1136: "GNSS Server",
    0x1137: "3D Display",
    0x1138: "3D Glasses",
    0x1139: "3D Synchronization",
    0x113a: "MPS Profile",
    0x113b: "MPS Service",
    /* 0x113c to 0x11ff undefined */
    0x1200: "PnP Information",
    0x1201: "Generic Networking",
    0x1202: "Generic File Transfer",
    0x1203: "Generic Audio",
    0x1204: "Generic Telephony",
    0x1205: "UPNP Service",
    0x1206: "UPNP IP Service",
    0x1300: "UPNP IP PAN",
    0x1301: "UPNP IP LAP",
    0x1302: "UPNP IP L2CAP",
    0x1303: "Video Source",
    0x1304: "Video Sink",
    0x1305: "Video Distribution",
    /* 0x1306 to 0x13ff undefined */
    0x1400: "HDP",
    0x1401: "HDP Source",
    0x1402: "HDP Sink",
    /* 0x1403 to 0x17ff undefined */
    0x1800: "Generic Access Profile",
    0x1801: "Generic Attribute Profile",
    0x1802: "Immediate Alert",
    0x1803: "Link Loss",
    0x1804: "Tx Power",
    0x1805: "Current Time Service",
    0x1806: "Reference Time Update Service",
    0x1807: "Next DST Change Service",
    0x1808: "Glucose",
    0x1809: "Health Thermometer",
    0x180a: "Device Information",
    /* 0x180b and 0x180c undefined */
    0x180d: "Heart Rate",
    0x180e: "Phone Alert Status Service",
    0x180f: "Battery Service",
    0x1810: "Blood Pressure",
    0x1811: "Alert Notification Service",
    0x1812: "Human Interface Device",
    0x1813: "Scan Parameters",
    0x1814: "Running Speed and Cadence",
    0x1815: "Automation IO",
    0x1816: "Cycling Speed and Cadence",
    /* 0x1817 undefined */
    0x1818: "Cycling Power",
    0x1819: "Location and Navigation",
    0x181a: "Environmental Sensing",
    0x181b: "Body Composition",
    0x181c: "User Data",
    0x181d: "Weight Scale",
    0x181e: "Bond Management",
    0x181f: "Continuous Glucose Monitoring",
    0x1820: "Internet Protocol Support",
    0x1821: "Indoor Positioning",
    0x1822: "Pulse Oximeter",
    0x1823: "HTTP Proxy",
    0x1824: "Transport Discovery",
    0x1825: "Object Transfer",
    /* 0x1824 to 0x27ff undefined */
    0x2800: "Primary Service",
    0x2801: "Secondary Service",
    0x2802: "Include",
    0x2803: "Characteristic",
    /* 0x2804 to 0x28ff undefined */
    0x2900: "Characteristic Extended Properties",
    0x2901: "Characteristic User Description",
    0x2902: "Client Characteristic Configuration",
    0x2903: "Server Characteristic Configuration",
    0x2904: "Characteristic Format",
    0x2905: "Characteristic Aggregate Format",
    0x2906: "Valid Range",
    0x2907: "External Report Reference",
    0x2908: "Report Reference",
    0x2909: "Number of Digitals",
    0x290a: "Value Trigger Setting",
    0x290b: "Environmental Sensing Configuration",
    0x290c: "Environmental Sensing Measurement",
    0x290d: "Environmental Sensing Trigger Setting",
    0x290e: "Time Trigger Setting",
    /* 0x290f to 0x29ff undefined */
    0x2a00: "Device Name",
    0x2a01: "Appearance",
    0x2a02: "Peripheral Privacy Flag",
    0x2a03: "Reconnection Address",
    0x2a04: "Peripheral Preferred Connection Parameters",
    0x2a05: "Service Changed",
    0x2a06: "Alert Level",
    0x2a07: "Tx Power Level",
    0x2a08: "Date Time",
    0x2a09: "Day of Week",
    0x2a0a: "Day Date Time",
    /* 0x2a0b undefined */
    0x2a0c: "Exact Time 256",
    0x2a0d: "DST Offset",
    0x2a0e: "Time Zone",
    0x2a0f: "Local Time Information",
    /* 0x2a10 undefined */
    0x2a11: "Time with DST",
    0x2a12: "Time Accuracy",
    0x2a13: "Time Source",
    0x2a14: "Reference Time Information",
    /* 0x2a15 undefined */
    0x2a16: "Time Update Control Point",
    0x2a17: "Time Update State",
    0x2a18: "Glucose Measurement",
    0x2a19: "Battery Level",
    /* 0x2a1a and 0x2a1b undefined */
    0x2a1c: "Temperature Measurement",
    0x2a1d: "Temperature Type",
    0x2a1e: "Intermediate Temperature",
    /* 0x2a1f and 0x2a20 undefined */
    0x2a21: "Measurement Interval",
    0x2a22: "Boot Keyboard Input Report",
    0x2a23: "System ID",
    0x2a24: "Model Number String",
    0x2a25: "Serial Number String",
    0x2a26: "Firmware Revision String",
    0x2a27: "Hardware Revision String",
    0x2a28: "Software Revision String",
    0x2a29: "Manufacturer Name String",
    0x2a2a: "IEEE 11073-20601 Regulatory Cert. Data List",
    0x2a2b: "Current Time",
    0x2a2c: "Magnetic Declination",
    /* 0x2a2d to 0x2a30 undefined */
    0x2a31: "Scan Refresh",
    0x2a32: "Boot Keyboard Output Report",
    0x2a33: "Boot Mouse Input Report",
    0x2a34: "Glucose Measurement Context",
    0x2a35: "Blood Pressure Measurement",
    0x2a36: "Intermediate Cuff Pressure",
    0x2a37: "Heart Rate Measurement",
    0x2a38: "Body Sensor Location",
    0x2a39: "Heart Rate Control Point",
    /* 0x2a3a to 0x2a3e undefined */
    0x2a3f: "Alert Status",
    0x2a40: "Ringer Control Point",
    0x2a41: "Ringer Setting",
    0x2a42: "Alert Category ID Bit Mask",
    0x2a43: "Alert Category ID",
    0x2a44: "Alert Notification Control Point",
    0x2a45: "Unread Alert Status",
    0x2a46: "New Alert",
    0x2a47: "Supported New Alert Category",
    0x2a48: "Supported Unread Alert Category",
    0x2a49: "Blood Pressure Feature",
    0x2a4a: "HID Information",
    0x2a4b: "Report Map",
    0x2a4c: "HID Control Point",
    0x2a4d: "Report",
    0x2a4e: "Protocol Mode",
    0x2a4f: "Scan Interval Window",
    0x2a50: "PnP ID",
    0x2a51: "Glucose Feature",
    0x2a52: "Record Access Control Point",
    0x2a53: "RSC Measurement",
    0x2a54: "RSC Feature",
    0x2a55: "SC Control Point",
    0x2a56: "Digital",
    /* 0x2a57 undefined */
    0x2a58: "Analog",
    0x2a59: "Analog Output",
    0x2a5a: "Aggregate",
    0x2a5b: "CSC Measurement",
    0x2a5c: "CSC Feature",
    0x2a5d: "Sensor Location",
    /* 0x2a5e to 0x2a62 undefined */
    0x2a63: "Cycling Power Measurement",
    0x2a64: "Cycling Power Vector",
    0x2a65: "Cycling Power Feature",
    0x2a66: "Cycling Power Control Point",
    0x2a67: "Location and Speed",
    0x2a68: "Navigation",
    0x2a69: "Position Quality",
    0x2a6a: "LN Feature",
    0x2a6b: "LN Control Point",
    0x2a6c: "Elevation",
    0x2a6d: "Pressure",
    0x2a6e: "Temperature",
    0x2a6f: "Humidity",
    0x2a70: "True Wind Speed",
    0x2a71: "True Wind Direction",
    0x2a72: "Apparent Wind Speed",
    0x2a73: "Apparent Wind Direction",
    0x2a74: "Gust Factor",
    0x2a75: "Pollen Concentration",
    0x2a76: "UV Index",
    0x2a77: "Irradiance",
    0x2a78: "Rainfall",
    0x2a79: "Wind Chill",
    0x2a7a: "Heat Index",
    0x2a7b: "Dew Point",
    0x2a7c: "Trend",
    0x2a7d: "Descriptor Value Changed",
    0x2a7e: "Aerobic Heart Rate Lower Limit",
    0x2a7f: "Aerobic Threshold",
    0x2a80: "Age",
    0x2a81: "Anaerobic Heart Rate Lower Limit",
    0x2a82: "Anaerobic Heart Rate Upper Limit",
    0x2a83: "Anaerobic Threshold",
    0x2a84: "Aerobic Heart Rate Upper Limit",
    0x2a85: "Date of Birth",
    0x2a86: "Date of Threshold Assessment",
    0x2a87: "Email Address",
    0x2a88: "Fat Burn Heart Rate Lower Limit",
    0x2a89: "Fat Burn Heart Rate Upper Limit",
    0x2a8a: "First Name",
    0x2a8b: "Five Zone Heart Rate Limits",
    0x2a8c: "Gender",
    0x2a8d: "Heart Rate Max",
    0x2a8e: "Height",
    0x2a8f: "Hip Circumference",
    0x2a90: "Last Name",
    0x2a91: "Maximum Recommended Heart Rate",
    0x2a92: "Resting Heart Rate",
    0x2a93: "Sport Type for Aerobic/Anaerobic Thresholds",
    0x2a94: "Three Zone Heart Rate Limits",
    0x2a95: "Two Zone Heart Rate Limit",
    0x2a96: "VO2 Max",
    0x2a97: "Waist Circumference",
    0x2a98: "Weight",
    0x2a99: "Database Change Increment",
    0x2a9a: "User Index",
    0x2a9b: "Body Composition Feature",
    0x2a9c: "Body Composition Measurement",
    0x2a9d: "Weight Measurement",
    0x2a9e: "Weight Scale Feature",
    0x2a9f: "User Control Point",
    0x2aa0: "Magnetic Flux Density - 2D",
    0x2aa1: "Magnetic Flux Density - 3D",
    0x2aa2: "Language",
    0x2aa3: "Barometric Pressure Trend",
    0x2aa4: "Bond Management Control Point",
    0x2aa5: "Bond Management Feature",
    0x2aa6: "Central Address Resolution",
    0x2aa7: "CGM Measurement",
    0x2aa8: "CGM Feature",
    0x2aa9: "CGM Status",
    0x2aaa: "CGM Session Start Time",
    0x2aab: "CGM Session Run Time",
    0x2aac: "CGM Specific Ops Control Point",
    0x2aad: "Indoor Positioning Configuration",
    0x2aae: "Latitude",
    0x2aaf: "Longitude",
    0x2ab0: "Local North Coordinate",
    0x2ab1: "Local East Coordinate",
    0x2ab2: "Floor Number",
    0x2ab3: "Altitude",
    0x2ab4: "Uncertainty",
    0x2ab5: "Location Name",
    0x2ab6: "URI",
    0x2ab7: "HTTP Headers",
    0x2ab8: "HTTP Status Code",
    0x2ab9: "HTTP Entity Body",
    0x2aba: "HTTP Control Point",
    0x2abb: "HTTPS Security",
    0x2abc: "TDS Control Point",
    0x2abd: "OTS Feature",
    0x2abe: "Object Name",
    0x2abf: "Object Type",
    0x2ac0: "Object Size",
    0x2ac1: "Object First-Created",
    0x2ac2: "Object Last-Modified",
    0x2ac3: "Object ID",
    0x2ac4: "Object Properties",
    0x2ac5: "Object Action Control Point",
    0x2ac6: "Object List Control Point",
    0x2ac7: "Object List Filter",
    0x2ac8: "Object Changed",
    /* vendor defined */
    0xfeff: "GN Netcom",
    0xfefe: "GN ReSound A/S",
    0xfefd: "Gimbal, Inc.",
    0xfefc: "Gimbal, Inc.",
    0xfefb: "Telit Wireless Solutions",
    0xfefa: "PayPal, Inc.",
    0xfef9: "PayPal, Inc.",
    0xfef8: "Aplix Corporation",
    0xfef7: "Aplix Corporation",
    0xfef6: "Wicentric, Inc.",
    0xfef5: "Dialog Semiconductor GmbH",
    0xfef4: "Google Inc.",
    0xfef3: "Google Inc.",
    0xfef2: "CSR",
    0xfef1: "CSR",
    0xfef0: "Intel",
    0xfeef: "Polar Electro Oy",
    0xfeee: "Polar Electro Oy",
    0xfeed: "Tile, Inc.",
    0xfeec: "Tile, Inc.",
    0xfeeb: "Swirl Networks, Inc.",
    0xfeea: "Swirl Networks, Inc.",
    0xfee9: "Quintic Corp.",
    0xfee8: "Quintic Corp.",
    0xfee7: "Tencent Holdings Limited",
    0xfee6: "Silvair, Inc.",
    0xfee5: "Nordic Semiconductor ASA",
    0xfee4: "Nordic Semiconductor ASA",
    0xfee3: "Anki, Inc.",
    0xfee2: "Anki, Inc.",
    0xfee1: "Anhui Huami Information Technology Co.",
    0xfee0: "Anhui Huami Information Technology Co.",
    0xfedf: "Design SHIFT",
    0xfede: "Coin, Inc.",
    0xfedd: "Jawbone",
    0xfedc: "Jawbone",
    0xfedb: "Perka, Inc.",
    0xfeda: "ISSC Technologies Corporation",
    0xfed9: "Pebble Technology Corporation",
    0xfed8: "Google Inc.",
    0xfed7: "Broadcom Corporation",
    0xfed6: "Broadcom Corporation",
    0xfed5: "Plantronics Inc.",
    0xfed4: "Apple, Inc.",
    0xfed3: "Apple, Inc.",
    0xfed2: "Apple, Inc.",
    0xfed1: "Apple, Inc.",
    0xfed0: "Apple, Inc.",
    0xfecf: "Apple, Inc.",
    0xfece: "Apple, Inc.",
    0xfecd: "Apple, Inc.",
    0xfecc: "Apple, Inc.",
    0xfecb: "Apple, Inc.",
    0xfeca: "Apple, Inc.",
    0xfec9: "Apple, Inc.",
    0xfec8: "Apple, Inc.",
    0xfec7: "Apple, Inc.",
    0xfec6: "Kocomojo, LLC",
    0xfec5: "Realtek Semiconductor Corp.",
    0xfec4: "PLUS Location Systems",
    0xfec3: "360fly, Inc.",
    0xfec2: "Blue Spark Technologies, Inc.",
    0xfec1: "KDDI Corporation",
    0xfec0: "KDDI Corporation",
    0xfebf: "Nod, Inc.",
    0xfebe: "Bose Corporation",
    0xfebd: "Clover Network, Inc.",
    0xfebc: "Dexcom, Inc.",
    0xfebb: "adafruit industries",
    0xfeba: "Tencent Holdings Limited",
    0xfeb9: "LG Electronics",
    0xfeb8: "Facebook, Inc.",
    0xfeb7: "Facebook, Inc.",
    0xfeb6: "Vencer Co, Ltd",
    0xfeb5: "WiSilica Inc.",
    0xfeb4: "WiSilica Inc.",
    0xfeb3: "Taobao",
    0xfeb2: "Microsoft Corporation",
    0xfeb1: "Electronics Tomorrow Limited",
    0xfeb0: "Nest Labs Inc.",
    0xfeaf: "Nest Labs Inc.",
    0xfeae: "Nokia Corporation",
    0xfead: "Nokia Corporation",
    0xfeac: "Nokia Corporation",
    0xfeab: "Nokia Corporation",
    0xfeaa: "Google Inc.",
    0xfea9: "Savant Systems LLC",
    0xfea8: "Savant Systems LLC",
    0xfea7: "UTC Fire and Security",
    0xfea6: "GoPro, Inc.",
    0xfea5: "GoPro, Inc.",
    0xfea4: "Paxton Access Ltd",
    0xfea3: "ITT Industries",
    0xfea2: "Intrepid Control Systems, Inc.",
    0xfea1: "Intrepid Control Systems, Inc.",
    0xfea0: "Google Inc.",
    0xfe9f: "Google Inc.",
    0xfe9e: "Dialog Semiconductor B.V.",
    0xfe9d: "Mobiquity Networks Inc",
    0xfe9c: "GSI Laboratories, Inc.",
    0xfe9b: "Samsara Networks, Inc",
    0xfe9a: "Estimote",
    0xfe99: "Currant, Inc.",
    0xfe98: "Currant, Inc.",
    0xfe97: "Tesla Motor Inc.",
    0xfe96: "Tesla Motor Inc.",
    0xfe95: "Xiaomi Inc.",
    0xfe94: "OttoQ Inc.",
    0xfe93: "OttoQ Inc.",
    0xfe92: "Jarden Safety & Security",
    0xfe91: "Shanghai Imilab Technology Co.,Ltd",
    0xfe90: "JUMA",
    0xfe8f: "CSR",
    0xfe8e: "ARM Ltd",
    0xfe8d: "Interaxon Inc.",
    0xfe8c: "TRON Forum",
    0xfe8b: "Apple, Inc.",
    0xfe8a: "Apple, Inc.",
    0xfe89: "B&O Play A/S",
    0xfe88: "SALTO SYSTEMS S.L.",
    0xfe87: "Qingdao Yeelink Information Technology Co., Ltd. ( 青岛亿联客信息技术有限公司 )",
    0xfe86: "HUAWEI Technologies Co., Ltd. ( 华为技术有限公司 )",
    0xfe85: "RF Digital Corp",
    0xfe84: "RF Digital Corp",
    0xfe83: "Blue Bite",
    0xfe82: "Medtronic Inc.",
    0xfe81: "Medtronic Inc.",
    0xfe80: "Doppler Lab",
    0xfe7f: "Doppler Lab",
    0xfe7e: "Awear Solutions Ltd",
    0xfe7d: "Aterica Health Inc.",
    0xfe7c: "Stollmann E+V GmbH",
    0xfe7b: "Orion Labs, Inc.",
    0xfe7a: "Bragi GmbH",
    0xfe79: "Zebra Technologies",
    0xfe78: "Hewlett-Packard Company",
    0xfe77: "Hewlett-Packard Company",
    0xfe76: "TangoMe",
    0xfe75: "TangoMe",
    0xfe74: "unwire",
    0xfe73: "St. Jude Medical, Inc.",
    0xfe72: "St. Jude Medical, Inc.",
    0xfe71: "Plume Design Inc",
    0xfe70: "Beijing Jingdong Century Trading Co., Ltd.",
    0xfe6f: "LINE Corporation",
    0xfe6e: "The University of Tokyo",
    0xfe6d: "The University of Tokyo",
    0xfe6c: "TASER International, Inc.",
    0xfe6b: "TASER International, Inc.",
    0xfe6a: "Kontakt Micro-Location Sp. z o.o.",
    0xfe69: "Qualcomm Life Inc",
    0xfe68: "Qualcomm Life Inc",
    0xfe67: "Lab Sensor Solutions",
    0xfe66: "Intel Corporation",
    0xfe65: "CHIPOLO d.o.o.",
    0xfe64: "Siemens AG",
    0xfe63: "Connected Yard, Inc.",
    0xfe62: "Indagem Tech LLC",
    0xfe61: "Logitech International SA",
    0xfe60: "Lierda Science & Technology Group Co., Ltd.",
    0xfe5f: "Eyefi, Inc.",
    0xfe5e: "Plastc Corporation",
    0xfe5d: "Grundfos A/S",
    0xfe5c: "million hunters GmbH",
    0xfe5b: "GT-tronics HK Ltd",
    0xfe5a: "Chronologics Corporation",
    0xfe59: "Nordic Semiconductor ASA",
    0xfe58: "Nordic Semiconductor ASA",
    0xfe57: "Dotted Labs",
    0xfe56: "Google Inc.",
    0xfe55: "Google Inc.",
    0xfe54: "Motiv, Inc.",
    0xfe53: "3M",
    0xfe52: "SetPoint Medical",
    0xfe51: "SRAM",
    0xfe50: "Google Inc.",
    0xfe4f: "Molekule, Inc.",
    0xfe4e: "NTT docomo",
    0xfe4d: "Casambi Technologies Oy",
    0xfe4c: "Volkswagen AG",
    0xfe4b: "Philips Lighting B.V.",
    0xfe4a: "OMRON HEALTHCARE Co., Ltd.",
    0xfe49: "SenionLab AB",
    0xfe48: "General Motors",
    0xfe47: "General Motors",
    0xfe46: "B&O Play A/S",
    0xfe45: "Snapchat Inc",
    0xfe44: "SK Telecom",
    0xfe43: "Andreas Stihl AG & Co. KG",
    0xfe42: "Nets A/S",
    0xfe41: "Inugo Systems Limited",
    0xfe40: "Inugo Systems Limited",
    0xfe3f: "Friday Labs Limited",
    0xfe3e: "BD Medical",
    0xfe3d: "BD Medical",
    0xfe3c: "Alibaba",
    0xfe3b: "Dolby Laboratories",
    0xfe3a: "TTS Tooltechnic Systems AG & Co.",
    0xfe39: "TTS Tooltechnic Systems AG & Co.",
    0xfe38: "Spaceek LTD",
    0xfe37: "Spaceek LTD",
    0xfe36: "HUAWEI Technologies Co., Ltd",
    0xfe35: "HUAWEI Technologies Co., Ltd",
    0xfe34: "SmallLoop LLC",
    0xfe33: "CHIPOLO d.o.o.",
    0xfe32: "Pro-Mark, Inc.",
    0xfe31: "Volkswagen AG",
    0xfe30: "Volkswagen AG",
    0xfe2f: "CRESCO Wireless, Inc",
    0xfe2e: "ERi,Inc.",
    0xfe2d: "SMART INNOVATION Co.,Ltd",
    0xfe2c: "Google Inc.",
    0xfe2b: "ITT Industries",
    0xfe2a: "DaisyWorks, Inc.",
    0xfe29: "Gibson Innovations",
    0xfe28: "Ayla Networks",
    0xfe27: "Google Inc.",
    0xfe26: "Google Inc.",
    0xfe25: "Apple, Inc.",
    0xfe24: "August Home Inc",
    0xfe23: "Zoll Medical Corporation",
    0xfe22: "Zoll Medical Corporation",
    0xfe21: "Bose Corporation",
    0xfe20: "Emerson",
    0xfe1f: "Garmin International, Inc.",
    0xfe1e: "Smart Innovations Co., Ltd",
    0xfe1d: "Illuminati Instrument Corporation",
    0xfe1c: "NetMedia, Inc.",
    0xfe1b: "Tyto Life LLC",
    0xfe1a: "Tyto Life LLC",
    0xfe19: "Google Inc.",
    0xfe18: "Runtime, Inc.",
    0xfe17: "Telit Wireless Solutions GmbH",
    0xfe16: "Footmarks, Inc.",
    0xfe15: "Amazon.com Services, Inc.",
    0xfe14: "Flextronics International USA Inc.",
    0xfe13: "Apple Inc.",
    0xfe12: "M-Way Solutions GmbH",
    0xfe11: "GMC-I Messtechnik GmbH",
    0xfe10: "Lapis Semiconductor Co., Ltd.",
    0xfe0f: "Philips Lighting B.V.",
    0xfe0e: "Setec Pty Ltd",
    0xfe0d: "Procter & Gamble",
    0xfe0c: "Procter & Gamble",
    0xfe0b: "ruwido austria gmbh",
    0xfe0a: "ruwido austria gmbh",
    0xfe09: "Pillsy, Inc.",
    0xfe08: "Microsoft",
    0xfe06: "Qualcomm Technologies, Inc.",
    0xfe05: "CORE Transport Technologies NZ Limited",
    0xfe04: "OpenPath Security Inc",
    0xfe03: "Amazon.com Services, Inc.",
    0xfe02: "Robert Bosch GmbH",
    0xfe01: "Duracell U.S. Operations Inc.",
    0xfe00: "Amazon.com Services, Inc.",
    0xfdff: "OSRAM GmbH",
    0xfdfe: "ADHERIUM(NZ) LIMITED",
    0xfdfd: "RecursiveSoft Inc.",
    0xfdfc: "Optrel AG",
    0xfdfb: "Tandem Diabetes Care",
    0xfdfa: "Tandem Diabetes Care",
    0xfdf9: "INIA",
    0xfdf8: "Onvocal",
    0xfdf7: "HP Inc.",
    0xfdf6: "AIAIAI ApS",
    0xfdf5: "Milwaukee Electric Tools",
    0xfdf4: "O. E. M. Controls, Inc.",
    0xfdf3: "Amersports",
    0xfdf2: "AMICCOM Electronics Corporation",
    0xfdf1: "LAMPLIGHT Co.,Ltd",
    0xfdf0: "Google Inc.",
    0xfdef: "ART AND PROGRAM, INC.",
    0xfdee: "Huawei Technologies Co., Ltd.",
    0xfded: "Pole Star",
    0xfdec: "Mannkind Corporation",
    0xfdeb: "Syntronix Corporation",
    0xfdea: "SeeScan, Inc",
    0xfde9: "Spacesaver Corporation",
    0xfde8: "Robert Bosch GmbH",
    0xfde7: "SECOM Co., LTD",
    0xfde6: "Intelletto Technologies Inc",
    0xfde5: "SMK Corporation",
    0xfde4: "JUUL Labs, Inc.",
    0xfde3: "Abbott Diabetes Care",
    /* SDO defined */
    0xfffe: "Alliance for Wireless Power (A4WP)",
    0xfffd: "Fast IDentity Online Alliance (FIDO)"
]
