#!/usr/bin/env bash
#
# Usage: ./removeUnneededSymbols.sh StreamChatSwiftUI ./Products
#
# Creating an xcframework for StreamChatSwiftUI generates a .bcsymbolmap file for itself, and one for
# each of its dependencies too (eg. StreamChat). That means that we will end up having something like:
#
# -> StreamChatSwiftUI/BCSymbolMaps/
#                              <UUID-StreamChatSwiftUI>.bcsymbolmap
#                              <UUID-StreamChat>.bcsymbolmap
#
# When adding both StreamChat and StreamChatSwiftUI to an app, it will throw an error when trying to compile
# saying that there are multiple executions producing the same file (<UUID-StreamChat>.bcsymbolmap).
#
# This script will remove duplicated .bcsymbolmap in the generated xcframeworks.
# If we countinue with the same example, it will leave it as follows:
#
# -> StreamChatSwiftUI/BCSymbolMaps/
#                              <UUID-StreamChatSwiftUI>.bcsymbolmap
#
# Each xcframework only contains its symbols now.

args=("$@")
library=$1
output_directory=$2

function removeUnneededSymbols() {
	arch=$1
	current_directory=$(pwd)

	cd "$output_directory/$library.xcframework/$arch/BCSymbolMaps"

	# Looking for [...]/DerivedSources/[LIBRARY-NAME]_vers.c
	regex="(\/DerivedSources\/)([a-zA-Z_]*)(_vers.c)"
	files="*.bcsymbolmap"
	for f in $files
	do
		text=`head -10 $f`
	    [[ $text =~ $regex ]]
	    library_match="${BASH_REMATCH[2]}"
	    if [[ $library_match != $library ]]
	    then
	    	echo "→ Removing uneeded 'bcsymbolmap' from $library-$arch: $library_match - $f"
	    	rm $f
	    fi
	done

	cd "$current_directory"
}

removeUnneededSymbols "ios-arm64"
removeUnneededSymbols "ios-arm64_x86_64-simulator"
