import {
  getInput,
  getBooleanInput,
  setFailed,
  info,
  error,
  warning,
  setOutput,
  startGroup,
  endGroup
} from '@actions/core'
import got from 'got'
import { readFileSync } from 'fs'
import { join, normalize } from 'path'
import semverDiff from 'semver-diff'

const semverRE =
  /^(?<major>0|[1-9]\d*)\.(?<minor>0|[1-9]\d*)\.(?<patch>0|[1-9]\d*)(?:-(?<prerelease>(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+(?<buildmetadata>[0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?$/

const semverReGlobal =
  /(?<major>0|[1-9]\d*)\.(?<minor>0|[1-9]\d*)\.(?<patch>0|[1-9]\d*)(?:-(?<prerelease>(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+(?<buildmetadata>[0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?/gm

const packageFileName = normalize(getInput('file-name') || 'package.json'),
  dir = process.env.GITHUB_WORKSPACE || '/github/workspace',
  eventFile = process.env.GITHUB_EVENT_PATH || '/github/workflow/event.json',
  assumeSameVersion = getInput('assume-same-version') as
    | 'old'
    | 'new'
    | undefined,
  staticChecking = getInput('static-checking') as
    | 'localIsNew'
    | 'remoteIsNew'
    | undefined,
  token = getInput('token')

let packageFileURL = (getInput('file-url') || '').trim()
const allowedTags = ['::before']

type outputKey = 'changed' | 'type' | 'version' | 'commit'
const outputs: Partial<Record<outputKey, string | boolean>> = {}

// #region Functions
async function main() {
  if (
    packageFileURL &&
    !isURL(packageFileURL) &&
    !allowedTags.includes(packageFileURL)
  )
    return setFailed(
      `The provided package file URL is not valid (received: ${packageFileURL})`
    )
  if (assumeSameVersion && !['old', 'new'].includes(assumeSameVersion))
    return setFailed(
      `The provided assume-same-version parameter is not valid (received ${assumeSameVersion})`
    )
  if (staticChecking && !['localIsNew', 'remoteIsNew'].includes(staticChecking))
    return setFailed(
      `The provided static-checking parameter is not valid (received ${staticChecking})`
    )

  const isPackageFileURLBefore = packageFileURL === '::before'

  if (isPackageFileURLBefore) {
    const event = await readJson(eventFile)
    if (!event) throw new Error(`Can't find event file (${eventFile})`)

    const { before, repository } = event
    if (before && repository) {
      packageFileURL = `https://raw.githubusercontent.com/${repository?.full_name}/${before}/${packageFileName}`
      startGroup('URL tag resolution...')
      info(
        `::before tag resolved to ${repository?.full_name}/${String(
          before
        ).substr(0, 7)}/${packageFileName}`
      )
      info(`Current package file URL: ${packageFileURL}`)
      info(`Using token for remote url: ${!!token}`)
      endGroup()
    } else
      throw new Error(
        `Can't correctly read event file (before: ${before}, repository: ${repository})`
      )
  }

  if (staticChecking) {
    if (!packageFileURL)
      return setFailed(
        'Static checking cannot be performed without a `file-url` argument.'
      )

    startGroup('Static-checking files...')
    info(`Package file name: "${packageFileName}"`)
    info(`Package file URL: "${packageFileURL}"`)
    const local: string = (await readJson(join(dir, packageFileName)))?.version,
      remote: string = (
        await readJson(
          packageFileURL,
          (isPackageFileURLBefore || isRawGithubUrl(packageFileURL)) && token
            ? token
            : undefined
        )
      )?.version
    if (!local || !remote) {
      endGroup()
      return setFailed(`Couldn't find ${local ? 'remote' : 'local'} version.`)
    }

    if (!semverRE.test(local)) {
      return setFailed(`Local version does not match semver pattern`)
    }

    if (!semverRE.test(remote)) {
      return setFailed(`Remote version does not match semver pattern`)
    }

    const versionDiff =
      staticChecking === 'localIsNew'
        ? semverDiff(remote, local)
        : semverDiff(local, remote)

    if (versionDiff) {
      output('changed', true)
      output('version', staticChecking == 'localIsNew' ? local : remote)
      output('type', versionDiff)

      endGroup()
      info(
        `Found match for version ${
          staticChecking == 'localIsNew' ? local : remote
        }`
      )
    }
  } else {
    const eventObj = await readJson(eventFile)
    const commits =
      eventObj.commits ||
      (await request(eventObj.pull_request._links.commits.href))
    await processDirectory(dir, commits)
  }
}

function isURL(str: string) {
  try {
    new URL(str)
    return true
  } catch {
    return false
  }
}

function isRawGithubUrl(str: string) {
  const url = new URL(str)
  return url.hostname === 'raw.githubusercontent.com'
}

async function readJson(file: string, token?: string) {
  if (isURL(file)) {
    const headers = token
      ? {
          Authorization: `token ${token}`
        }
      : {}
    return (
      await got({
        url: file,
        method: 'GET',
        headers,
        responseType: 'json'
      })
    ).body
  } else {
    const data = readFileSync(file, { encoding: 'utf8' })
    if (typeof data == 'string')
      try {
        return JSON.parse(data)
      } catch (e) {
        error(e instanceof Error ? e.stack || e.message : e + '')
      }
  }
}

async function request(url: string) {
  const headers = token
    ? {
        Authorization: `Bearer ${token}`
      }
    : {}
  return (
    await got({
      url,
      method: 'GET',
      headers,
      responseType: 'json'
    })
  ).body
}

async function processDirectory(
  dir: string,
  commits: LocalCommit[] | PartialCommitResponse[]
) {
  try {
    const packageObj = await (
      packageFileURL
        ? readJson(packageFileURL)
        : readJson(join(dir, packageFileName))
    ).catch(() => {
      Promise.reject(
        new NeutralExitError(`Package file not found: ${packageFileName}`)
      )
    })

    if (!isPackageObj(packageObj)) throw new Error("Can't find version field")

    if (commits.length >= 20)
      warning(
        'This workflow run topped the commit limit set by GitHub webhooks: that means that commits could not appear and that the run could not find the version change.'
      )

    if (commits.length <= 0) {
      info('There are no commits to look at.')
      return
    }

    await checkCommits(commits, packageObj.version)
  } catch (e) {
    setFailed(`${e}`)
  }
}

async function checkCommits(
  commits: LocalCommit[] | PartialCommitResponse[],
  version: string
) {
  try {
    startGroup(
      `Searching in ${commits.length} commit${
        commits.length == 1 ? '' : 's'
      }...`
    )
    info(`Package file name: "${packageFileName}"`)
    info(
      `Package file URL: ${
        packageFileURL ? `"${packageFileURL}"` : 'undefined'
      }`
    )
    info(
      `Version assumptions: ${
        assumeSameVersion ? `"${assumeSameVersion}"` : 'undefined'
      }`
    )
    for (const commit of commits) {
      const { message, sha } = getBasicInfo(commit)
      const match: string[] = message.match(semverReGlobal) || []
      if (match.includes(version)) {
        if (await checkDiff(sha, version)) {
          endGroup()
          info(
            `Found match for version ${version}: ${sha.substring(
              0,
              7
            )} ${message}`
          )
          return true
        }
      }
    }
    endGroup()

    if (getBooleanInput('diff-search')) {
      info(
        'No standard npm version commit found, switching to diff search (this could take more time...)'
      )

      if (!isLocalCommitArray(commits)) {
        commits = commits.sort(
          (a, b) =>
            new Date(b.commit.committer.date).getTime() -
            new Date(a.commit.committer.date).getTime()
        )
      }

      startGroup(
        `Checking the diffs of ${commits.length} commit${
          commits.length == 1 ? '' : 's'
        }...`
      )
      for (const commit of commits) {
        const { message, sha } = getBasicInfo(commit)

        if (await checkDiff(sha, version)) {
          endGroup()
          info(
            `Found match for version ${version}: ${sha.substring(
              0,
              7
            )} - ${message}`
          )
          return true
        }
      }
    }

    endGroup()
    info('No matching commit found.')
    output('changed', false)
    return false
  } catch (e) {
    setFailed(`${e}`)
  }
}

function getBasicInfo(commit: LocalCommit | PartialCommitResponse) {
  let message: string, sha: string

  if (isLocalCommit(commit)) {
    message = commit.message
    sha = commit.id
  } else {
    message = commit.commit.message
    sha = commit.sha
  }

  return {
    message,
    sha
  }
}

async function checkDiff(sha: string, version: string) {
  try {
    const commit = await getCommit(sha)
    const pkg = commit.files.find((f) => f.filename == packageFileName)
    if (!pkg) {
      info(`- ${sha.substr(0, 7)}: no changes to the package file`)
      return false
    }

    const versionLines: {
      added?: string
      deleted?: string
    } = {}

    const rawLines = pkg.patch
      .split('\n')
      .filter(
        (line) => line.includes('"version":') && ['+', '-'].includes(line[0])
      )
    if (rawLines.length > 2) {
      info(`- ${sha.substr(0, 7)}: too many version lines`)
      return false
    }

    for (const line of rawLines)
      versionLines[line.startsWith('+') ? 'added' : 'deleted'] = line
    if (!versionLines.added) {
      info(`- ${sha.substr(0, 7)}: no "+ version" line`)
      return false
    }

    const versions = {
      added:
        assumeSameVersion == 'new'
          ? version
          : parseVersionLine(versionLines.added),
      deleted:
        assumeSameVersion == 'old'
          ? version
          : !!versionLines.deleted && parseVersionLine(versionLines.deleted)
    }
    if (versions.added != version && !assumeSameVersion) {
      info(
        `- ${sha.substr(
          0,
          7
        )}: added version doesn't match current one (added: "${
          versions.added
        }"; current: "${version}")`
      )
      return false
    }

    output('changed', true)
    output('version', version)
    if (versions.deleted)
      output('type', semverDiff(versions.deleted, versions.added as string))
    output('commit', commit.sha)

    info(`- ${sha.substr(0, 7)}: match found, more info below`)
    return true
  } catch (e) {
    error(`An error occurred in checkDiff:\n${e}`)
    throw new ExitError(1)
  }
}

function trimTrailingSlashFromUrl(rawUrl) {
  const url = rawUrl.trim()
  return url.endsWith('/') ? url.slice(0, -1) : url
}

async function getCommit(sha: string): Promise<CommitResponse> {
  const githubApiUrl = trimTrailingSlashFromUrl(getInput('github-api-url'))
  const url = `${githubApiUrl}/repos/${process.env.GITHUB_REPOSITORY}/commits/${sha}`
  const res = await request(url)

  if (typeof res != 'object' || !res)
    throw new Error('Response data must be an object.')

  return res as CommitResponse
}

function parseVersionLine(str: string) {
  return (str.split('"') || []).map((s) => matchVersion(s)).find((e) => !!e)
}

function matchVersion(str: string): string {
  return (str.match(semverReGlobal) || ([] as string[]))[0]
}

function output(name: outputKey, value?: string | boolean) {
  outputs[name] = value
  return setOutput(name, `${value}`)
}

function logOutputs() {
  startGroup('Outputs:')
  for (const key in outputs) {
    info(`${key}: ${outputs[key]}`)
  }
  endGroup()
}
// #endregion

// #region Error classes
class ExitError extends Error {
  code?: number

  constructor(code: number | null) {
    super(`Command failed with code ${code}`)
    if (typeof code == 'number') this.code = code
  }
}

class NeutralExitError extends Error {}
// #endregion

if (require.main == module) {
  info('Searching for version update...')
  main()
    .then(() => {
      if (typeof outputs.changed == 'undefined') {
        output('changed', false)
      }

      logOutputs()
    })
    .catch((e) => {
      if (e instanceof NeutralExitError) process.exitCode = 78
      else {
        process.exitCode = 1
        error(e.message || e)
      }
    })
}

// #region Types and interfaces
interface CommitResponse extends PartialCommitResponse {
  files: {
    filename: string
    additions: number
    deletions: number
    changes: number
    status: string
    raw_url: string
    blob_url: string
    patch: string
  }[]
}

interface LocalCommit {
  id: string
  message: string
  author: {
    name: string
    email: string
  }
  url: string
  distinct: boolean
}
function isLocalCommit(value): value is LocalCommit {
  return typeof value.id == 'string'
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function isLocalCommitArray(value: any[]): value is LocalCommit[] {
  return isLocalCommit(value[0])
}

interface PackageObj {
  version: string
}
function isPackageObj(value): value is PackageObj {
  return !!value && !!value.version
}

interface PartialCommitResponse {
  url: string
  sha: string
  node_id: string
  html_url: string
  comments_url: string
  commit: {
    url: string
    author: {
      name: string
      email: string
      date: string
    }
    committer: {
      name: string
      email: string
      date: string
    }
    message: string
    tree: {
      url: string
      sha: string
    }
    comment_count: number
    verification: {
      verified: boolean
      reason: string
      signature: object | null
      payload: object | null
    }
  }
  author: {
    login: string
    id: number
    node_id: string
    avatar_url: string
    gravatar_id: string
    url: string
    html_url: string
    followers_url: string
    following_url: string
    gists_url: string
    starred_url: string
    subscriptions_url: string
    organizations_url: string
    repos_url: string
    events_url: string
    receive_events_url: string
    type: string
    site_admin: boolean
  }
  committer: {
    login: string
    id: number
    node_id: string
    avatar_url: string
    gravatar_id: string
    url: string
    html_url: string
    followers_url: string
    following_url: string
    gists_url: string
    starred_url: string
    subscriptions_url: string
    organizations_url: string
    repos_url: string
    events_url: string
    receive_events_url: string
    type: string
    site_admin: boolean
  }
  parents: {
    url: string
    sha: string
  }[]
  stats: {
    additions: number
    deletions: number
    total: number
  }
}
// #endregion
