//
//  DevCycleOptions.swift
//  DevCycle
//

import Foundation

public class DevCycleOptions {
    private(set) public var eventFlushIntervalMS: Int?
    private(set) public var disableEventLogging: Bool?
    public var logLevel: LogLevel = .error
    private(set) public var enableEdgeDB: Bool = false
    private(set) public var disableConfigCache: Bool = false
    private(set) public var configCacheTTL: Int = 604_800_000
    private(set) public var disableRealtimeUpdates: Bool = false
    private(set) public var disableAutomaticEventLogging: Bool = false
    private(set) public var disableCustomEventLogging: Bool = false
    private(set) public var apiProxyURL: String?
    private(set) public var eventsApiProxyURL: String?

    public class OptionsBuilder {
        var options: DevCycleOptions

        init() {
            self.options = DevCycleOptions()
        }

        // Controls the interval between flushing events to the DevCycle servers in milliseconds, defaults to 10 seconds.
        public func eventFlushIntervalMS(_ interval: Int? = 10000) -> OptionsBuilder {
            self.options.eventFlushIntervalMS = interval
            return self
        }

        @available(*, deprecated, message: "Use eventFlushIntervalMS")
        public func flushEventsIntervalMs(_ interval: Int? = 10000) -> OptionsBuilder {
            self.options.eventFlushIntervalMS = interval
            return self
        }

        @available(
            *, deprecated, message: "Use disableAutomaticEventLogging or disableCustomEventLogging"
        )
        public func disableEventLogging(_ disable: Bool) -> OptionsBuilder {
            self.options.disableEventLogging = disable
            return self
        }

        // Disables logging of SDK generated events (e.g. variableEvaluated, variableDefaulted) to DevCycle.
        public func disableAutomaticEventLogging(_ disable: Bool) -> OptionsBuilder {
            self.options.disableAutomaticEventLogging = disable
            return self
        }

        // Disables logging of custom events generated by calling .track() method to DevCycle.
        public func disableCustomEventLogging(_ disable: Bool) -> OptionsBuilder {
            self.options.disableCustomEventLogging = disable
            return self
        }

        // Controls the log level of the SDK, defaults to `error`
        public func logLevel(_ level: LogLevel) -> OptionsBuilder {
            self.options.logLevel = level
            return self
        }

        // Enables the usage of EdgeDB for DevCycle that syncs User Data to DevCycle.
        public func enableEdgeDB(_ enable: Bool) -> OptionsBuilder {
            self.options.enableEdgeDB = enable
            return self
        }

        // Disable the use of cached configs
        public func disableConfigCache(_ disable: Bool) -> OptionsBuilder {
            self.options.disableConfigCache = disable
            return self
        }

        // The maximum allowed age of a cached config in milliseconds, defaults to 7 days
        public func configCacheTTL(_ ttl: Int = 604_800_000) -> OptionsBuilder {
            self.options.configCacheTTL = ttl
            return self
        }

        // Disable Realtime Update and their SSE connection.
        public func disableRealtimeUpdates(_ disable: Bool) -> OptionsBuilder {
            self.options.disableRealtimeUpdates = disable
            return self
        }

        // Allows the SDK to communicate with a proxy of DevCycle APIs.
        public func apiProxyURL(_ proxyURL: String) -> OptionsBuilder {
            self.options.apiProxyURL = proxyURL
            return self
        }

        // Allows the SDK to use a proxy to send to for the events API
        public func eventsApiProxyURL(_ proxyURL: String) -> OptionsBuilder {
            self.options.eventsApiProxyURL = proxyURL
            return self
        }

        public func build() -> DevCycleOptions {
            let result = self.options
            self.options = DevCycleOptions()
            return result
        }
    }

    public static func builder() -> OptionsBuilder {
        return OptionsBuilder()
    }
}

@available(*, deprecated, message: "Use DevCycleOptions")
public typealias DVCOptions = DevCycleOptions
