/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-Present Datadog, Inc.
 *
 * This file includes software developed by Flight School, https://flight.school/ and altered by Datadog.
 * Use of this source code is governed by MIT license:
 *
 * Copyright 2018 Read Evaluate Press, LLC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

import XCTest
import DatadogInternal

class AnyDecodableTests: XCTestCase {
    func testJSONDecoding() throws {
        let json = """
        {
            "boolean": true,
            "integer": 42,
            "double": 3.141592653589793,
            "string": "string",
            "array": [1, 2, 3],
            "nested": {
                "a": "alpha",
                "b": "bravo",
                "c": "charlie"
            },
            "null": null
        }
        """.data(using: .utf8)!

        let decoder = JSONDecoder()
        let dictionary = try decoder.decode([String: AnyDecodable].self, from: json)

        XCTAssertEqual(dictionary["boolean"]?.value as? Bool, true)
        XCTAssertEqual(dictionary["integer"]?.value as? Int, 42)
        XCTAssertEqual(dictionary["double"]?.value as! Double, 3.141592653589793, accuracy: 0.001)
        XCTAssertEqual(dictionary["string"]?.value as? String, "string")
        XCTAssertEqual(dictionary["array"]?.value as? [Int], [1, 2, 3])
        XCTAssertEqual(dictionary["nested"]?.value as? [String: String], ["a": "alpha", "b": "bravo", "c": "charlie"])
        XCTAssertEqual(dictionary["null"]?.value as? NSNull, NSNull())
    }

    func testAnyDecoding() throws {
        class Passthrough: PassthroughAnyCodable {
            init() {}
        }

        let passthrough = Passthrough()

        let any: [String: Any?] = [
            "boolean": true,
            "integer": 42,
            "double": 3.141592653589793,
            "string": "string",
            "array": [1, 2, 3],
            "nested": [
                "a": "alpha",
                "b": "bravo",
                "c": "charlie"
            ],
            "null": nil,
            "uuid": UUID(),
            "url": URL(string: "https://test.com"),
            "passthrough": passthrough
        ]

        let decoder = AnyDecoder()
        let dictionary = try decoder.decode([String: AnyDecodable].self, from: any)

        XCTAssertEqual(dictionary["boolean"]?.value as? Bool, true)
        XCTAssertEqual(dictionary["integer"]?.value as? Int, 42)
        XCTAssertEqual(dictionary["double"]?.value as! Double, 3.141592653589793, accuracy: 0.001)
        XCTAssertEqual(dictionary["string"]?.value as? String, "string")
        XCTAssertEqual(dictionary["array"]?.value as? [Int], [1, 2, 3])
        XCTAssertEqual(dictionary["nested"]?.value as? [String: String], ["a": "alpha", "b": "bravo", "c": "charlie"])
        XCTAssert(dictionary["uuid"]?.value is UUID)
        XCTAssertEqual(dictionary["url"]?.value as? URL, URL(string: "https://test.com"))
        XCTAssert(dictionary["passthrough"]?.value as? Passthrough === passthrough)
    }

    func testAnyDecodingFailue() throws {
        class NotPassthrough {
            init() {}
        }

        let passthrough = NotPassthrough()

        let any: [String: Any?] = [
            "passthrough": passthrough
        ]

        let decoder = AnyDecoder()
        XCTAssertThrowsError(try decoder.decode(AnyDecodable.self, from: any)) { error in
            guard case DecodingError.dataCorrupted(let context) = error else {
                return XCTFail("Unexpected error: \(error)")
            }

            XCTAssertEqual(context.debugDescription, "AnyDecodable value cannot be decoded")
        }
    }
}
