/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-Present Datadog, Inc.
 */

import XCTest
@testable import DatadogCore

final class EventGeneratorTests: XCTestCase {
    func testEmpty() throws {
        let dataBlocks = [BatchDataBlock]()

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 0)
    }

    func testWithoutEvent() throws {
        let dataBlocks = [BatchDataBlock(type: .eventMetadata, data: Data([0x01]))]

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 0)
    }

    func testEventWithoutMetadata() throws {
        let dataBlocks = [BatchDataBlock(type: .event, data: Data([0x01]))]

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 1)
        XCTAssertEqual(events[0].data, Data([0x01]))
        XCTAssertNil(events[0].metadata)
    }

    func testEventWithMetadata() throws {
        let dataBlocks = [
            BatchDataBlock(type: .eventMetadata, data: Data([0x02])),
            BatchDataBlock(type: .event, data: Data([0x01]))
        ]

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 1)
        XCTAssertEqual(events[0].metadata, Data([0x02]))
        XCTAssertEqual(events[0].data, Data([0x01]))
    }

    func testEventWithCurruptedMetadata() throws {
        let dataBlocks = [
            BatchDataBlock(type: .eventMetadata, data: Data([0x03])), // skipped from reading
            BatchDataBlock(type: .eventMetadata, data: Data([0x02])),
            BatchDataBlock(type: .event, data: Data([0x01]))
        ]

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 1)
        XCTAssertEqual(events[0].metadata, Data([0x02]))
        XCTAssertEqual(events[0].data, Data([0x01]))
    }

    func testEvents() {
        let dataBlocks = [
            BatchDataBlock(type: .eventMetadata, data: Data([0x02])),
            BatchDataBlock(type: .event, data: Data([0x01])),
            BatchDataBlock(type: .event, data: Data([0x03])),
            BatchDataBlock(type: .event, data: Data([0x05]))
        ]

        let sut = EventGenerator(dataBlocks: dataBlocks)
        let events = sut.map { $0 }
        XCTAssertEqual(events.count, 3)

        XCTAssertEqual(events[0].data, Data([0x01]))
        XCTAssertEqual(events[0].metadata, Data([0x02]))

        XCTAssertEqual(events[1].data, Data([0x03]))
        XCTAssertNil(events[1].metadata)

        XCTAssertEqual(events[2].data, Data([0x05]))
        XCTAssertNil(events[2].metadata)
    }
}
