/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-2020 Datadog, Inc.
 */

import Foundation
import DatadogInternal

internal struct DistributedTracing {
    /// Tracing sampler used to sample traces generated by the SDK.
    let sampler: Sampler
    /// The distributed tracing ID generator.
    let traceIDGenerator: TraceIDGenerator
    let spanIDGenerator: SpanIDGenerator
    /// First party hosts defined by the user.
    let firstPartyHosts: FirstPartyHosts
    /// Trace context injection configuration to determine whether the trace context should be injected or not.
    let traceContextInjection: TraceContextInjection

    init(
        sampler: Sampler,
        firstPartyHosts: FirstPartyHosts,
        traceIDGenerator: TraceIDGenerator,
        spanIDGenerator: SpanIDGenerator,
        traceContextInjection: TraceContextInjection
    ) {
        self.sampler = sampler
        self.traceIDGenerator = traceIDGenerator
        self.spanIDGenerator = spanIDGenerator
        self.firstPartyHosts = firstPartyHosts
        self.traceContextInjection = traceContextInjection
    }
}

internal final class URLSessionRUMResourcesHandler: DatadogURLSessionHandler, RUMCommandPublisher {
    /// The date provider
    let dateProvider: DateProvider
    /// DistributinTracing
    let distributedTracing: DistributedTracing?
    /// Attributes-providing callback.
    /// It is configured by the user and should be used to associate additional RUM attributes with intercepted RUM Resource.
    let rumAttributesProvider: RUM.ResourceAttributesProvider?

    /// First party hosts defined by the user.
    var firstPartyHosts: FirstPartyHosts {
        distributedTracing?.firstPartyHosts ?? .init()
    }

    // MARK: - Initialization

    init(
        dateProvider: DateProvider,
        rumAttributesProvider: RUM.ResourceAttributesProvider?,
        distributedTracing: DistributedTracing?
    ) {
        self.dateProvider = dateProvider
        self.rumAttributesProvider = rumAttributesProvider
        self.distributedTracing = distributedTracing
    }

    // MARK: - Internal

    weak var subscriber: RUMCommandSubscriber?

    func publish(to subscriber: RUMCommandSubscriber) {
        self.subscriber = subscriber
    }

    // MARK: - DatadogURLSessionHandler

    func modify(request: URLRequest, headerTypes: Set<DatadogInternal.TracingHeaderType>) -> (URLRequest, TraceContext?) {
        distributedTracing?.modify(request: request, headerTypes: headerTypes) ?? (request, nil)
    }

    func interceptionDidStart(interception: DatadogInternal.URLSessionTaskInterception) {
        let url = interception.request.url?.absoluteString ?? "unknown_url"
        interception.register(origin: "rum")

        subscriber?.process(
            command: RUMStartResourceCommand(
                resourceKey: interception.identifier.uuidString,
                time: dateProvider.now,
                attributes: [:],
                url: url,
                httpMethod: RUMMethod(httpMethod: interception.request.httpMethod),
                kind: RUMResourceType(request: interception.request.unsafeOriginal),
                spanContext: distributedTracing?.trace(from: interception)
            )
        )
    }

    func interceptionDidComplete(interception: DatadogInternal.URLSessionTaskInterception) {
        guard let subscriber = subscriber else {
            return DD.logger.warn(
                """
                RUM Resource was completed, but no `RUMMonitor` is initialized in the core. RUM auto instrumentation will not work.
                Make sure `RUMMonitor.initialize()` is called before any network request is send.
                """
            )
        }

        // Get RUM Resource attributes from the user.
        let userAttributes = rumAttributesProvider?(
            interception.request.unsafeOriginal,
            interception.completion?.httpResponse,
            interception.data,
            interception.completion?.error
        ) ?? [:]

        if let resourceMetrics = interception.metrics {
            subscriber.process(
                command: RUMAddResourceMetricsCommand(
                    resourceKey: interception.identifier.uuidString,
                    time: dateProvider.now,
                    attributes: [:],
                    metrics: resourceMetrics
                )
            )
        }

        if let httpResponse = interception.completion?.httpResponse {
            subscriber.process(
                command: RUMStopResourceCommand(
                    resourceKey: interception.identifier.uuidString,
                    time: dateProvider.now,
                    attributes: userAttributes,
                    kind: RUMResourceType(response: httpResponse),
                    httpStatusCode: httpResponse.statusCode,
                    size: interception.metrics?.responseSize
                )
            )
        }

        if let error = interception.completion?.error {
            subscriber.process(
                command: RUMStopResourceWithErrorCommand(
                    resourceKey: interception.identifier.uuidString,
                    time: dateProvider.now,
                    error: error,
                    source: .network,
                    httpStatusCode: interception.completion?.httpResponse?.statusCode,
                    attributes: userAttributes
                )
            )
        }
    }
}

extension DistributedTracing {
    func modify(request: URLRequest, headerTypes: Set<DatadogInternal.TracingHeaderType>) -> (URLRequest, TraceContext?) {
        let traceID = traceIDGenerator.generate()
        let spanID = spanIDGenerator.generate()
        let injectedSpanContext = TraceContext(
            traceID: traceID,
            spanID: spanID,
            parentSpanID: nil,
            sampleRate: sampler.samplingRate,
            isKept: sampler.sample()
        )

        var request = request
        var hasSetAnyHeader = false
        headerTypes.forEach {
            let writer: TracePropagationHeadersWriter
            switch $0 {
            case .datadog:
                writer = HTTPHeadersWriter(
                    samplingStrategy: .headBased,
                    traceContextInjection: traceContextInjection
                )
                // To make sure the generated traces from RUM don’t affect APM Index Spans counts.
                request.setValue("rum", forHTTPHeaderField: TracingHTTPHeaders.originField)
            case .b3:
                writer = B3HTTPHeadersWriter(
                    samplingStrategy: .headBased,
                    injectEncoding: .single,
                    traceContextInjection: traceContextInjection
                )
            case .b3multi:
                writer = B3HTTPHeadersWriter(
                    samplingStrategy: .headBased,
                    injectEncoding: .multiple,
                    traceContextInjection: traceContextInjection
                )
            case .tracecontext:
                writer = W3CHTTPHeadersWriter(
                    samplingStrategy: .headBased,
                    tracestate: [
                        W3CHTTPHeaders.Constants.origin: W3CHTTPHeaders.Constants.originRUM
                    ],
                    traceContextInjection: traceContextInjection
                )
            }

            writer.write(traceContext: injectedSpanContext)

            writer.traceHeaderFields.forEach { field, value in
                // do not overwrite existing header
                if request.value(forHTTPHeaderField: field) == nil {
                    hasSetAnyHeader = true
                    request.setValue(value, forHTTPHeaderField: field)
                }
            }
        }

        return (request, (hasSetAnyHeader && injectedSpanContext.isKept) ? injectedSpanContext : nil)
    }

    func trace(from interception: DatadogInternal.URLSessionTaskInterception) -> RUMSpanContext? {
        return interception.trace.map {
            .init(
                traceID: $0.traceID,
                spanID: $0.spanID,
                samplingRate: Double(sampler.samplingRate) / 100.0
            )
        }
    }
}

private extension HTTPURLResponse {
    func asClientError() -> Error? {
        // 4xx Client Errors
        guard statusCode >= 400 && statusCode < 500 else {
            return nil
        }
        let message = "\(statusCode) " + HTTPURLResponse.localizedString(forStatusCode: statusCode)
        return NSError(domain: "HTTPURLResponse", code: statusCode, userInfo: [NSLocalizedDescriptionKey: message])
    }
}
