/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-2020 Datadog, Inc.
 */

import UIKit

internal class UIViewControllerSwizzler {
    let viewDidAppear: ViewDidAppear
    let viewDidDisappear: ViewDidDisappear

    init(handler: UIKitRUMViewsHandlerType) throws {
        self.viewDidAppear = try ViewDidAppear(handler: handler)
        self.viewDidDisappear = try ViewDidDisappear(handler: handler)
    }

    func swizzle() {
        viewDidAppear.swizzle()
        viewDidDisappear.swizzle()
    }

    // MARK: - Swizzlings

    /// Swizzles the `UIViewController.viewDidAppear()`
    class ViewDidAppear: MethodSwizzler <
        @convention(c) (UIViewController, Selector, Bool) -> Void,
        @convention(block) (UIViewController, Bool) -> Void
    > {
        private static let selector = #selector(UIViewController.viewDidAppear(_:))
        private let method: FoundMethod
        private let handler: UIKitRUMViewsHandlerType

        init(handler: UIKitRUMViewsHandlerType) throws {
            self.method = try Self.findMethod(with: Self.selector, in: UIViewController.self)
            self.handler = handler
        }

        func swizzle() {
            typealias Signature = @convention(block) (UIViewController, Bool) -> Void
            swizzle(method) { previousImplementation -> Signature in
                return { [weak handler = self.handler] vc, animated  in
                    handler?.notify_viewDidAppear(viewController: vc, animated: animated)
                    previousImplementation(vc, Self.selector, animated)
                }
            }
        }
    }

    /// Swizzles the `UIViewController.viewDidDisappear()`
    class ViewDidDisappear: MethodSwizzler <
        @convention(c) (UIViewController, Selector, Bool) -> Void,
        @convention(block) (UIViewController, Bool) -> Void
    > {
        private static let selector = #selector(UIViewController.viewDidDisappear(_:))
        private let method: FoundMethod
        private let handler: UIKitRUMViewsHandlerType

        init(handler: UIKitRUMViewsHandlerType) throws {
            self.method = try Self.findMethod(with: Self.selector, in: UIViewController.self)
            self.handler = handler
        }

        func swizzle() {
            typealias Signature = @convention(block) (UIViewController, Bool) -> Void
            swizzle(method) { previousImplementation -> Signature in
                return { [weak handler = self.handler] vc, animated  in
                    handler?.notify_viewDidDisappear(viewController: vc, animated: animated)
                    previousImplementation(vc, Self.selector, animated)
                }
            }
        }
    }
}
