/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-Present Datadog, Inc.
 */

import Foundation
import CoreGraphics
import UIKit

/// The `ViewTreeSnapshot` is an intermediate representation of the app UI in Session Replay
/// recording: [views hierarchy] → [`ViewTreeSnapshot`] → [wireframes].
///
/// Although it's being built from the actual views hierarchy, it doesn't correspond 1:1 to it. Similarly,
/// it doesn't translate 1:1 into wireframes that get uploaded to the SR BE. Instead, it provides its
/// own description of the view hierarchy, which can be optimised for efficiency in SR recorder (e.g. unlike
/// the real views hierarchy, `ViewTreeSnapshot` is meant to be safe when accesed on a background thread).
internal struct ViewTreeSnapshot {
    /// The time of taking this snapshot.
    let date: Date
    /// The RUM context from the moment of taking this snapshot.
    let rumContext: RUMContext
    /// The size of a viewport in this snapshot.
    let viewportSize: CGSize
    /// An array of nodes recorded for this snapshot - sequenced in DFS order.
    let nodes: [Node]
}

/// An individual node in `ViewTreeSnapshot`. A `Node` describes a single view - similar: an array of nodes describes
/// view and its subtree (in depth-first order).
///
/// Typically, to describe certain view-tree we need significantly less nodes than number of views, because some views
/// are meaningless for session replay (e.g. hidden views or containers with no appearance).
///
/// **Note:** The purpose of this structure is to be lightweight and create minimal overhead when the view-tree
/// is captured on the main thread (the `Recorder` constantly creates `Nodes` for views residing in the hierarchy).
internal struct Node {
    /// Attributes of the `UIView` that this node was created for.
    let viewAttributes: ViewAttributes

    /// The semantics of this node.
    let semantics: NodeSemantics
}

/// Attributes of the `UIView` that the node was created for.
///
/// It is used by the `Recorder` to capture view attributes on the main thread.
/// It enforces immutability for later (thread safe) access from background queue in `Processor`.
internal struct ViewAttributes: Equatable {
    /// The view's `frame`, in VTS's root view's coordinate space (usually, the screen coordinate space).
    let frame: CGRect

    /// Original view's `.backgorundColor`.
    let backgroundColor: CGColor?

    /// Original view's `layer.borderColor`.
    let layerBorderColor: CGColor?

    /// Original view's `layer.borderWidth`.
    let layerBorderWidth: CGFloat

    /// Original view's `layer.cornerRadius`.
    let layerCornerRadius: CGFloat

    /// Original view's `.alpha` (between `0.0` and `1.0`).
    let alpha: CGFloat

    /// Original view's `.isHidden`.
    let isHidden: Bool

    /// Original view's `.intrinsicContentSize`.
    let intrinsicContentSize: CGSize

    /// If the view is technically visible (different than `!isHidden` because it also considers `alpha` and `frame != .zero`).
    /// A view can be technically visible, but it may have no appearance in practise (e.g. if its colors use `0` alpha component).
    ///
    /// Example 1: A view is invisible if it has `.zero` size or it is fully transparent (`alpha == 0`).
    /// Example 2: A view can be visible if it has fully transparent background color, but its `alpha` is `0.5` or it occupies non-zero area.
    var isVisible: Bool { !isHidden && alpha > 0 && frame != .zero }

    /// If the view has any visible appearance (considering: background color + border style).
    /// In other words: if this view brings anything visual.
    ///
    /// Example: A view might have no appearance if it has `0` border width and transparent fill color.
    var hasAnyAppearance: Bool {
        let borderAlpha = layerBorderColor?.alpha ?? 0
        let hasBorderAppearance = layerBorderWidth > 0 && borderAlpha > 0

        let fillAlpha = backgroundColor?.alpha ?? 0
        let hasFillAppearance = fillAlpha > 0

        return isVisible && (hasBorderAppearance || hasFillAppearance)
    }

    /// If the view is translucent, meaining if any content underneath it can be seen.
    ///
    /// Example: A view with with blue background of alpha `0.5` is considered "translucent".
    var isTranslucent: Bool { !isVisible || alpha < 1 }
}

extension ViewAttributes {
    init(frameInRootView: CGRect, view: UIView) {
        self.frame = frameInRootView
        self.backgroundColor = view.backgroundColor?.cgColor
        self.layerBorderColor = view.layer.borderColor
        self.layerBorderWidth = view.layer.borderWidth
        self.layerCornerRadius = view.layer.cornerRadius
        self.alpha = view.alpha
        self.isHidden = view.isHidden
        self.intrinsicContentSize = view.intrinsicContentSize
    }
}

/// A type denoting semantics of given UI element in Session Replay.
///
/// The `NodeSemantics` is attached to each node produced by `Recorder`. During tree traversal,
/// views are queried in available node recorders. Each `NodeRecorder` inspects the view object and
/// tries to infer its identity (a `NodeSemantics`).
///
/// There are two `NodeSemantics` that describe the identity of UI element:
/// - `AmbiguousElement` - element is of `UIView` class and we only know its base attributes (the real identity could be ambiguous);
/// - `SpecificElement` - element is one of `UIView` subclasses and we know its specific identity along with set of subclass-specific
/// attributes (e.g. text in `UILabel` or the "on" / "off" state of `UISwitch` control).
///
/// Additionally, there are two utility semantics that control the processing of nodes in SR:
/// - `InvisibleElement` - element is either `UIView` or one of its known subclasses, but it has no visual appearance in SR, so it can
/// be safely ignored in `Recorder` or `Processor` (e.g. a `UILabel` with no text, no border and fully transparent color).
/// - `UnknownElement` - the element is of unknown kind, which could indicate an error during view tree traversal (e.g. working on
/// assumption that is not met).
///
/// Both `AmbiguousElement` and `SpecificElement` provide an implementation of `NodeWireframesBuilder` which describes
/// how to construct SR wireframes for UI elements they refer to. No builder is provided for `InvisibleElement` and `UnknownElement`.
internal protocol NodeSemantics {
    /// The severity of this semantic.
    ///
    /// While querying certain `view` with an array of supported `NodeRecorders` each recorder can spot different semantics of
    /// the same view. In that case, the semantics with higher `importance` takes precedence.
    static var importance: Int { get }

    /// Defines the strategy which `Recorder` should apply to subtree of this node.
    var subtreeStrategy: NodeSubtreeStrategy { get }

    /// A type defining how to build SR wireframes for the UI element this semantic was recorded for.
    var wireframesBuilder: NodeWireframesBuilder? { set get }
}

extension NodeSemantics {
    /// The severity of this semantic.
    ///
    /// While querying certain `view` with an array of supported `NodeRecorders` each recorder can spot different semantics of
    /// the same view. In that case, the semantics with higher `importance` takes precedence.
    var importance: Int { Self.importance }
}

/// Strategies for handling node's subtree by `Recorder`.
internal enum NodeSubtreeStrategy {
    /// Continue traversing subtree of this node to record nested nodes automatically.
    ///
    /// This strategy is particularly useful for semantics that do not make assumption on node's content (e.g. this strategy can be
    /// practical choice for `UITabBar` node to let the recorder automatically capture any labels, images or shapes that are displayed in it).
    case record
    /// Do not traverse subtree of this node and instead replace it (the subtree) with provided nodes.
    ///
    /// This strategy is useful for semantics that only partially describe certain elements and perform curated traversal of their subtree (e.g. it can be
    /// used for `UIPickerView` where we only traverse subtree to look for specific elements, like the text of the selected row).
    case replace(subtreeNodes: [Node])
    /// Do not enter the subtree of this node.
    ///
    /// This strategy should be used for semantics that fully describe certain elements (e.g. it doesn't make sense to traverse the subtree of `UISwitch`).
    case ignore
}

/// Semantics of an UI element that is of unknown kind. Receiving this semantics in `Processor` could indicate an error
/// in view-tree traversal performed in `Recorder` (e.g. working on assumption that is not met).
internal struct UnknownElement: NodeSemantics {
    static let importance: Int = .min
    var wireframesBuilder: NodeWireframesBuilder? = nil
    let subtreeStrategy: NodeSubtreeStrategy = .record

    /// Use `UnknownElement.constant` instead.
    private init () {}

    /// A constant value of `UnknownElement` semantics.
    static let constant = UnknownElement()
}

/// A semantics of an UI element that is either `UIView` or one of its known subclasses. This semantics mean that the element
/// has no visual appearance that can be presented in SR (e.g. a `UILabel` with no text, no border and fully transparent color).
/// Nodes with this semantics can be safely ignored in `Recorder` or in `Processor`.
internal struct InvisibleElement: NodeSemantics {
    static let importance: Int = 0
    var wireframesBuilder: NodeWireframesBuilder? = nil
    let subtreeStrategy: NodeSubtreeStrategy

    /// Use `InvisibleElement.constant` instead.
    private init () {
        self.subtreeStrategy = .ignore
    }

    init(subtreeStrategy: NodeSubtreeStrategy) {
        self.subtreeStrategy = subtreeStrategy
    }

    /// A constant value of `InvisibleElement` semantics with `subtreeStrategy: .ignore`.
    static let constant = InvisibleElement()
}

/// A semantics of an UI element that is of `UIView` type. This semantics mean that the element has visual appearance in SR, but
/// it will only utilize its base `UIView` attributes. The full identity of the node will remain ambiguous if not overwritten with `SpecificElement`.
///
/// The view-tree traversal algorithm will continue visiting the subtree of given `UIView` if it has `AmbiguousElement` semantics.
internal struct AmbiguousElement: NodeSemantics {
    static let importance: Int = 0
    var wireframesBuilder: NodeWireframesBuilder?
    let subtreeStrategy: NodeSubtreeStrategy = .record
}

/// A semantics of an UI element that is one of `UIView` subclasses. This semantics mean that we know its full identity along with set of
/// subclass-specific attributes that will be used to render it in SR (e.g. all base `UIView` attributes plus the text in `UILabel` or the
/// "on" / "off" state of `UISwitch` control).
internal struct SpecificElement: NodeSemantics {
    static let importance: Int = .max
    var wireframesBuilder: NodeWireframesBuilder?
    let subtreeStrategy: NodeSubtreeStrategy
}
