/*
 * Unless explicitly stated otherwise all files in this repository are licensed under the Apache License Version 2.0.
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2019-2020 Datadog, Inc.
 */

import Foundation

@testable import Datadog

internal final class DatadogCoreMock: Flushable {
    private var v1Features: [String: Any] = [:]

    var context: DatadogV1Context?

    init(context: DatadogV1Context? = .mockAny()) {
        self.context = context
    }

    /// Flush resgistered features.
    ///
    /// The method will also call `flush` on any `Flushable` registered
    /// feature.
    func flush() {
        all(Flushable.self).forEach { $0.flush() }
        v1Features = [:]
    }

    /// Gets all registered feature of a given type.
    ///
    /// - Parameter type: The desired feature type.
    /// - Returns: Array of feature.
    func all<T>(_ type: T.Type) -> [T] {
        v1Features.values.compactMap { $0 as? T }
    }
}

extension DatadogCoreMock: DatadogV1CoreProtocol {
    // MARK: V1 interface

    func register<T>(feature instance: T?) {
        let key = String(describing: T.self)
        v1Features[key] = instance
    }

    func feature<T>(_ type: T.Type) -> T? {
        let key = String(describing: T.self)
        return v1Features[key] as? T
    }

    func scope<T>(for featureType: T.Type) -> V1FeatureScope? {
        guard let context = context else {
            return nil
        }

        let key = String(describing: T.self)

        guard let feature = v1Features[key] as? V1Feature else {
            return nil
        }

        return DatadogCoreFeatureScope(
            context: context,
            storage: feature.storage
        )
    }
}

extension DatadogV1Context: AnyMockable {
    static func mockAny() -> DatadogV1Context {
        return mockWith()
    }

    static func mockWith(
        configuration: CoreConfiguration = .mockAny(),
        dependencies: CoreDependencies = .mockAny(),
        appVersionProvider: AppVersionProvider = .mockAny()
    ) -> DatadogV1Context {
        return DatadogV1Context(
            configuration: configuration,
            dependencies: dependencies,
            appVersionProvider: appVersionProvider
        )
    }
}
