import type { LayoutServerLoad } from './$types';
import { CONTENT_TYPES } from '$lib/utils/content';
import { existsSync } from 'fs';
import { resolve } from 'path';

export const prerender = true;

async function checkContentType(type: string): Promise<boolean> {
	try {
		await import(`$static/${type}.json`);
		return true;
	} catch {
		return false;
	}
}

function checkDocExists(): boolean {
	// Check if static/doc directory exists
	return existsSync(resolve('static', 'doc'));
}

export const load: LayoutServerLoad = async () => {
	try {
		// Dynamic import to handle missing file gracefully
		const metadata = await import('$static/metadata.json');

		// Check which content types exist - dynamically from CONTENT_TYPES
		const contentTypesArray = await Promise.all(
			CONTENT_TYPES.map(async (type) => [type, await checkContentType(type)])
		);
		const contentTypes = Object.fromEntries(contentTypesArray);

		// Special handling for doc - check if directory exists instead of JSON
		contentTypes.doc = checkDocExists();

		return {
			...metadata.default,
			metadataError: null,
			contentTypes
		};
	} catch {
		// metadata.json is missing or invalid - return minimal data to keep layout working
		const emptyContentTypes = Object.fromEntries(
			[...CONTENT_TYPES, 'doc'].map((type) => [type, false])
		);
		
		return {
			name: '',
			summary: '',
			description: '',
			project_url: [],
			metadataError:
				'Project metadata not found. Please ensure metadata.json exists in the static folder. Did you forget to run `ap build`?',
			contentTypes: emptyContentTypes
		};
	}
};