//
//  STTwitterOSRequest.m
//  STTwitterDemoOSX
//
//  Created by Nicolas Seriot on 20/02/14.
//  Copyright (c) 2014 Nicolas Seriot. All rights reserved.
//

#import "STTwitterOSRequest.h"
#import <Social/Social.h>
#import <Accounts/Accounts.h>
#if TARGET_OS_IPHONE
#import <Twitter/Twitter.h> // iOS 5
#endif
#import "STHTTPRequest.h"
#import "NSString+STTwitter.h"
#import "NSError+STTwitter.h"

typedef void (^completion_block_t)(NSObject<STTwitterRequestProtocol> *request, NSDictionary *requestHeaders, NSDictionary *responseHeaders, id response);
typedef void (^error_block_t)(NSObject<STTwitterRequestProtocol> *request, NSDictionary *requestHeaders, NSDictionary *responseHeaders, NSError *error);
typedef void (^upload_progress_block_t)(NSInteger bytesWritten, NSInteger totalBytesWritten, NSInteger totalBytesExpectedToWrite);
typedef void (^stream_block_t)(NSObject<STTwitterRequestProtocol> *request, NSData *data);

@interface STTwitterOSRequest ()
@property (nonatomic, copy) completion_block_t completionBlock;
@property (nonatomic, copy) error_block_t errorBlock;
@property (nonatomic, copy) upload_progress_block_t uploadProgressBlock;
@property (nonatomic, copy) stream_block_t streamBlock;
@property (nonatomic, retain) NSURLConnection *connection;
@property (nonatomic, retain) NSHTTPURLResponse *httpURLResponse; // only used with streaming API
@property (nonatomic, retain) NSMutableData *data; // only used with non-streaming API
@property (nonatomic, retain) ACAccount *account;
@property (nonatomic) NSInteger httpMethod;
@property (nonatomic, retain) NSDictionary *params;
@property (nonatomic, retain) NSString *baseURLString;
@property (nonatomic, retain) NSString *resource;
@property (nonatomic) NSTimeInterval timeoutInSeconds;
@end


@implementation STTwitterOSRequest

- (instancetype)initWithAPIResource:(NSString *)resource
                      baseURLString:(NSString *)baseURLString
                         httpMethod:(NSInteger)httpMethod
                         parameters:(NSDictionary *)params
                            account:(ACAccount *)account
                   timeoutInSeconds:(NSTimeInterval)timeoutInSeconds
                uploadProgressBlock:(void(^)(NSInteger bytesWritten, NSInteger totalBytesWritten, NSInteger totalBytesExpectedToWrite))uploadProgressBlock
                        streamBlock:(void(^)(NSObject<STTwitterRequestProtocol> *request, NSData *data))streamBlock
                    completionBlock:(void(^)(NSObject<STTwitterRequestProtocol> *request, NSDictionary *requestHeaders, NSDictionary *responseHeaders, id response))completionBlock
                         errorBlock:(void(^)(NSObject<STTwitterRequestProtocol> *request, NSDictionary *requestHeaders, NSDictionary *responseHeaders, NSError *error))errorBlock {
    
    NSAssert(completionBlock, @"completionBlock is missing");
    NSAssert(errorBlock, @"errorBlock is missing");
    
    self = [super init];
    
    self.resource = resource;
    self.baseURLString = baseURLString;
    self.httpMethod = httpMethod;
    self.params = params;
    self.account = account;
    self.completionBlock = completionBlock;
    self.errorBlock = errorBlock;
    self.uploadProgressBlock = uploadProgressBlock;
    self.streamBlock = streamBlock;
    self.timeoutInSeconds = timeoutInSeconds;
    
    return self;
}

- (NSURLRequest *)preparedURLRequest {
    NSString *postDataKey = [_params valueForKey:kSTPOSTDataKey];
    NSString *postDataFilename = [_params valueForKey:kSTPOSTMediaFileNameKey];
    NSData *mediaData = [_params valueForKey:postDataKey];
    
    NSMutableDictionary *paramsWithoutMedia = [_params mutableCopy];
    if(postDataKey) [paramsWithoutMedia removeObjectForKey:postDataKey];
    [paramsWithoutMedia removeObjectForKey:kSTPOSTDataKey];
    [paramsWithoutMedia removeObjectForKey:kSTPOSTMediaFileNameKey];
    
    NSString *urlString = [_baseURLString stringByAppendingString:_resource];
    NSURL *url = [NSURL URLWithString:urlString];
    
    id request = nil;
    
#if TARGET_OS_IPHONE && (__IPHONE_OS_VERSION_MIN_REQUIRED < __IPHONE_6_0)
    
    if (floor(NSFoundationVersionNumber) < NSFoundationVersionNumber_iOS_6_0) {
        TWRequestMethod method = (_httpMethod == 0) ? TWRequestMethodGET : TWRequestMethodPOST;
        request = [[TWRequest alloc] initWithURL:url parameters:paramsWithoutMedia requestMethod:method];
    } else {
        request = [SLRequest requestForServiceType:SLServiceTypeTwitter requestMethod:_httpMethod URL:url parameters:paramsWithoutMedia];
    }
    
#else
    request = [SLRequest requestForServiceType:SLServiceTypeTwitter requestMethod:_httpMethod URL:url parameters:paramsWithoutMedia];
#endif
    
    [request setAccount:_account];
    
    if(mediaData) {
        NSString *filename = postDataFilename ? postDataFilename : @"media.jpg";
        [request addMultipartData:mediaData withName:postDataKey type:@"application/octet-stream" filename:filename];
    }
    
    // we use NSURLConnection because SLRequest doesn't play well with the streaming API
    
    NSURLRequest *preparedURLRequest = nil;
#if TARGET_OS_IPHONE && (__IPHONE_OS_VERSION_MIN_REQUIRED < __IPHONE_6_0)
    if (floor(NSFoundationVersionNumber) < NSFoundationVersionNumber_iOS_6_0) {
        preparedURLRequest = [request signedURLRequest];
    } else {
        preparedURLRequest = [request preparedURLRequest];
    }
#else
    preparedURLRequest = [request preparedURLRequest];
#endif
    
    return preparedURLRequest;
}

- (void)startRequest {
    
    NSURLRequest *preparedURLRequest = [self preparedURLRequest];
    
    NSMutableURLRequest *mutablePreparedURLRequest = [preparedURLRequest mutableCopy];
    mutablePreparedURLRequest.timeoutInterval = _timeoutInSeconds;
    
    if (_connection) {
        [self cancel];
    }
    _connection = [NSURLConnection connectionWithRequest:mutablePreparedURLRequest delegate:self];
    
    [_connection start];
}

- (void)cancel {
    [_connection cancel];
    
    NSURLRequest *request = [_connection currentRequest];
    
    NSString *s = @"Connection was cancelled.";
    NSDictionary *userInfo = @{NSLocalizedDescriptionKey: s};
    NSError *error = [NSError errorWithDomain:NSStringFromClass([self class])
                                         code:kSTHTTPRequestCancellationError
                                     userInfo:userInfo];
    self.errorBlock(self, [self requestHeadersForRequest:request], [_httpURLResponse allHeaderFields], error);
}

- (NSDictionary *)requestHeadersForRequest:(id)request {
    
    if([request isKindOfClass:[NSURLRequest class]]) {
        return [request allHTTPHeaderFields];
    }
    
#if TARGET_OS_IPHONE &&  (__IPHONE_OS_VERSION_MIN_REQUIRED < __IPHONE_6_0)
    if (floor(NSFoundationVersionNumber) < NSFoundationVersionNumber_iOS_6_0) {
        return [[request signedURLRequest] allHTTPHeaderFields];
    } else {
        return [[request preparedURLRequest] allHTTPHeaderFields];
    }
#else
    return [[request preparedURLRequest] allHTTPHeaderFields];
#endif
}

#pragma mark NSURLConnectionDataDelegate

- (void)connection:(NSURLConnection *)connection didReceiveResponse:(NSURLResponse *)response {
    
    if([response isKindOfClass:[NSHTTPURLResponse class]] == NO) return;
    
    self.httpURLResponse = (NSHTTPURLResponse *)response;
    
    self.data = [NSMutableData data];
}

- (void)connection:(NSURLConnection *)connection didReceiveData:(NSData *)data {
    
    BOOL isStreaming = [[[[connection originalRequest] URL] host] rangeOfString:@"stream"].location != NSNotFound;
    
    if(isStreaming) {
        self.streamBlock(self, data);
    } else {
        [self.data appendData:data];
    }
}

- (void)connection:(NSURLConnection *)connection didFailWithError:(NSError *)error {
    
    NSURLRequest *request = [connection currentRequest];
    NSDictionary *requestHeaders = [request allHTTPHeaderFields];
    NSDictionary *responseHeaders = [_httpURLResponse allHeaderFields];
    
    self.errorBlock(self, requestHeaders, responseHeaders, error);
}

- (void)connectionDidFinishLoading:(NSURLConnection *)connection {
    
    NSURLRequest *request = [connection currentRequest];
    
    if(_data == nil) {
        self.errorBlock(self, [self requestHeadersForRequest:request], [_httpURLResponse allHeaderFields], nil);
        return;
    }
    
    NSError *error = [NSError st_twitterErrorFromResponseData:_data responseHeaders:[_httpURLResponse allHeaderFields] underlyingError:nil];
    
    if(error) {
        self.errorBlock(self, [self requestHeadersForRequest:request], [_httpURLResponse allHeaderFields], error);
        return;
    }
    
    NSError *jsonError = nil;
    id response = [NSJSONSerialization JSONObjectWithData:_data options:NSJSONReadingAllowFragments error:&jsonError];
    
    if(response == nil) {
        // eg. reverse auth response
        // oauth_token=xxx&oauth_token_secret=xxx&user_id=xxx&screen_name=xxx
        response = [[NSString alloc] initWithData:_data encoding:NSUTF8StringEncoding];
    }
    
    if(response) {
        self.completionBlock(self, [self requestHeadersForRequest:request], [_httpURLResponse allHeaderFields], response);
    } else {
        self.errorBlock(self, [self requestHeadersForRequest:request], [_httpURLResponse allHeaderFields], jsonError);
    }
}

- (void)connection:(NSURLConnection *)connection
   didSendBodyData:(NSInteger)bytesWritten
 totalBytesWritten:(NSInteger)totalBytesWritten
totalBytesExpectedToWrite:(NSInteger)totalBytesExpectedToWrite {
    if(self.uploadProgressBlock == nil) return;
    
    // avoid overcommit while posting big images, like 5+ MB
    dispatch_queue_t queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0);
    dispatch_async(queue, ^{
        self.uploadProgressBlock(bytesWritten, totalBytesWritten, totalBytesExpectedToWrite);
    });
}

@end
